<?php

namespace App\Http\Controllers;

use App\Models\MonthlyBudget;
use App\Models\FinanceCategory;
use App\Models\FinanceRecord;
use Illuminate\Http\Request;
use Carbon\Carbon;

class MonthlyBudgetController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $selectedMonth = $request->input('month', now()->format('Y-m'));

        // Create default budgets if not exists
        MonthlyBudget::createDefaultBudgets($user->id, $selectedMonth);

        // Get budgets with category info
        $budgets = MonthlyBudget::forUser($user->id)
            ->forMonth($selectedMonth)
            ->with('categoryInfo')
            ->get();

        // Update spent amounts
        foreach ($budgets as $budget) {
            $budget->updateSpentAmount();
        }

        // Calculate totals
        $totalBudget = $budgets->sum('budget_amount');
        $totalSpent = $budgets->sum('spent_amount');
        $totalRemaining = $totalBudget - $totalSpent;

        // Get available months
        $availableMonths = [];
        $currentMonth = now();
        for ($i = -3; $i <= 3; $i++) {
            $month = $currentMonth->copy()->addMonths($i);
            $availableMonths[] = [
                'value' => $month->format('Y-m'),
                'label' => $month->locale('th')->monthName . ' ' . $month->year,
            ];
        }

        return view('finance.budgets.index', compact(
            'budgets',
            'selectedMonth',
            'availableMonths',
            'totalBudget',
            'totalSpent',
            'totalRemaining'
        ));
    }

    public function create(Request $request)
    {
        $month = $request->input('month', now()->format('Y-m'));
        $categories = FinanceCategory::expense()->active()->ordered()->get();

        return view('finance.budgets.create', compact('month', 'categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'month' => 'required|date_format:Y-m',
            'budgets' => 'required|array',
            'budgets.*.category' => 'required|exists:finance_categories,slug',
            'budgets.*.amount' => 'required|numeric|min:0',
        ]);

        $user = auth()->user();
        $month = $request->month;

        foreach ($request->budgets as $budgetData) {
            if ($budgetData['amount'] > 0) {
                MonthlyBudget::updateOrCreate([
                    'user_id' => $user->id,
                    'month' => $month,
                    'category' => $budgetData['category'],
                ], [
                    'budget_amount' => $budgetData['amount'],
                ]);
            }
        }

        return redirect()->route('finance.budgets.index', ['month' => $month])
            ->with('success', 'บันทึกงบประมาณเรียบร้อยแล้ว');
    }

    public function edit(MonthlyBudget $budget)
    {
        // Check ownership
        if ($budget->user_id !== auth()->id()) {
            abort(403);
        }

        $categories = FinanceCategory::expense()->active()->ordered()->get();

        return view('finance.budgets.edit', compact('budget', 'categories'));
    }

    public function update(Request $request, MonthlyBudget $budget)
    {
        // Check ownership
        if ($budget->user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'budget_amount' => 'required|numeric|min:0',
        ]);

        $budget->update([
            'budget_amount' => $request->budget_amount,
        ]);

        return redirect()->route('finance.budgets.index', ['month' => $budget->month])
            ->with('success', 'อัปเดตงบประมาณเรียบร้อยแล้ว');
    }

    public function copyFromPreviousMonth(Request $request)
    {
        $user = auth()->user();
        $currentMonth = $request->input('month', now()->format('Y-m'));

        MonthlyBudget::copyFromPreviousMonth($user->id, $currentMonth);

        return redirect()->route('finance.budgets.index', ['month' => $currentMonth])
            ->with('success', 'คัดลอกงบประมาณจากเดือนที่แล้วเรียบร้อยแล้ว');
    }

    public function destroy(MonthlyBudget $budget)
    {
        // Check ownership
        if ($budget->user_id !== auth()->id()) {
            abort(403);
        }

        $month = $budget->month;
        $budget->delete();

        return redirect()->route('finance.budgets.index', ['month' => $month])
            ->with('success', 'ลบงบประมาณเรียบร้อยแล้ว');
    }

    public function bulkUpdate(Request $request)
    {
        $request->validate([
            'month' => 'required|date_format:Y-m',
            'budgets' => 'required|array',
            'budgets.*.id' => 'required|exists:monthly_budgets,id',
            'budgets.*.amount' => 'required|numeric|min:0',
        ]);

        $user = auth()->user();

        foreach ($request->budgets as $budgetData) {
            $budget = MonthlyBudget::find($budgetData['id']);
            
            // Check ownership
            if ($budget && $budget->user_id === $user->id) {
                $budget->update([
                    'budget_amount' => $budgetData['amount'],
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'อัปเดตงบประมาณเรียบร้อยแล้ว',
        ]);
    }
}