<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class MonthlyBudget extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'month',
        'category',
        'budget_amount',
        'spent_amount',
    ];

    protected $casts = [
        'budget_amount' => 'decimal:2',
        'spent_amount' => 'decimal:2',
    ];

    /**
     * Relationships
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function categoryInfo()
    {
        return $this->belongsTo(FinanceCategory::class, 'category', 'slug');
    }

    /**
     * Attributes
     */
    public function getRemainingAmountAttribute()
    {
        return $this->budget_amount - $this->spent_amount;
    }

    public function getUsagePercentageAttribute()
    {
        if ($this->budget_amount <= 0) return 0;
        return round(($this->spent_amount / $this->budget_amount) * 100, 1);
    }

    public function getStatusAttribute()
    {
        $percentage = $this->usage_percentage;
        
        if ($percentage > 100) return 'over';
        if ($percentage > 80) return 'warning';
        return 'good';
    }

    public function getIsOverBudgetAttribute()
    {
        return $this->spent_amount > $this->budget_amount;
    }

    /**
     * Scopes
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeForMonth($query, $month)
    {
        return $query->where('month', $month);
    }

    /**
     * Methods
     */
    public function updateSpentAmount()
    {
        $spent = FinanceRecord::forUser($this->user_id)
            ->expense()
            ->forMonth($this->month)
            ->byCategory($this->category)
            ->sum('amount');
        
        $this->update(['spent_amount' => $spent]);
        
        return $this;
    }

    public static function createDefaultBudgets($userId, $month)
    {
        $expenseCategories = FinanceCategory::expense()->active()->get();
        
        foreach ($expenseCategories as $category) {
            self::firstOrCreate([
                'user_id' => $userId,
                'month' => $month,
                'category' => $category->slug,
            ], [
                'budget_amount' => 0,
                'spent_amount' => 0,
            ]);
        }
    }

    public static function copyFromPreviousMonth($userId, $currentMonth)
    {
        $previousMonth = \Carbon\Carbon::parse($currentMonth)->subMonth()->format('Y-m');
        $previousBudgets = self::forUser($userId)->forMonth($previousMonth)->get();
        
        foreach ($previousBudgets as $prevBudget) {
            self::firstOrCreate([
                'user_id' => $userId,
                'month' => $currentMonth,
                'category' => $prevBudget->category,
            ], [
                'budget_amount' => $prevBudget->budget_amount,
                'spent_amount' => 0,
            ]);
        }
    }
}