<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class CheckTranslations extends Command
{
    protected $signature = 'i18n:check {--locale=} {--fix} {--strict}';
    protected $description = 'Check for missing translation keys and optionally fix them';

    public function handle()
    {
        $locales = $this->option('locale') 
            ? [$this->option('locale')] 
            : config('app.available_locales', ['th', 'en']);
        
        $fix = $this->option('fix');
        $strict = $this->option('strict');
        $hasErrors = false;

        foreach ($locales as $locale) {
            $this->info("Checking translations for locale: {$locale}");
            
            $missing = $this->findMissingKeys($locale);
            $unused = $this->findUnusedKeys($locale);
            
            if (!empty($missing)) {
                $hasErrors = true;
                $this->error("Missing keys for {$locale}:");
                foreach ($missing as $key) {
                    $this->line("  - {$key}");
                }
                
                if ($fix) {
                    $this->fixMissingKeys($locale, $missing);
                }
            }
            
            if (!empty($unused)) {
                $this->warn("Unused keys for {$locale}:");
                foreach ($unused as $key) {
                    $this->line("  - {$key}");
                }
            }
            
            if (empty($missing) && empty($unused)) {
                $this->info("✓ All translations are complete for {$locale}");
            }
        }
        
        if ($strict && $hasErrors) {
            $this->error('Translation check failed. Missing keys found.');
            return 1;
        }
        
        return 0;
    }
    
    private function findMissingKeys($locale)
    {
        $usedKeys = $this->scanForTranslationKeys();
        $existingKeys = $this->getExistingKeys($locale);
        
        return array_diff($usedKeys, $existingKeys);
    }
    
    private function findUnusedKeys($locale)
    {
        $usedKeys = $this->scanForTranslationKeys();
        $existingKeys = $this->getExistingKeys($locale);
        
        return array_diff($existingKeys, $usedKeys);
    }
    
    private function scanForTranslationKeys()
    {
        $keys = [];
        
        // Scan Blade files
        $bladeFiles = File::allFiles(resource_path('views'));
        foreach ($bladeFiles as $file) {
            $content = File::get($file);
            
            // Match __('key') and __("key")
            preg_match_all('/__\([\'"]([^\'"]+)[\'"]\)/', $content, $matches);
            $keys = array_merge($keys, $matches[1]);
            
            // Match {{ __('key') }}
            preg_match_all('/\{\{\s*__\([\'"]([^\'"]+)[\'"]\)\s*\}\}/', $content, $matches);
            $keys = array_merge($keys, $matches[1]);
            
            // Match @lang('key')
            preg_match_all('/@lang\([\'"]([^\'"]+)[\'"]\)/', $content, $matches);
            $keys = array_merge($keys, $matches[1]);
        }
        
        // Scan PHP files
        $phpFiles = File::allFiles(app_path());
        foreach ($phpFiles as $file) {
            $content = File::get($file);
            
            // Match __('key') and __("key")
            preg_match_all('/__\([\'"]([^\'"]+)[\'"]\)/', $content, $matches);
            $keys = array_merge($keys, $matches[1]);
            
            // Match trans('key')
            preg_match_all('/trans\([\'"]([^\'"]+)[\'"]\)/', $content, $matches);
            $keys = array_merge($keys, $matches[1]);
        }
        
        return array_unique($keys);
    }
    
    private function getExistingKeys($locale)
    {
        $keys = [];
        $langPath = resource_path("lang/{$locale}");
        
        if (File::exists($langPath)) {
            $files = File::files($langPath);
            
            foreach ($files as $file) {
                $filename = $file->getFilenameWithoutExtension();
                $content = include $file->getPathname();
                
                if (is_array($content)) {
                    $keys = array_merge($keys, $this->flattenArray($content, $filename));
                }
            }
        }
        
        // Load JSON translations
        $jsonFile = resource_path("lang/{$locale}.json");
        if (File::exists($jsonFile)) {
            $jsonContent = json_decode(File::get($jsonFile), true);
            if (is_array($jsonContent)) {
                $keys = array_merge($keys, array_keys($jsonContent));
            }
        }
        
        return array_unique($keys);
    }
    
    private function flattenArray($array, $prefix = '')
    {
        $result = [];
        
        foreach ($array as $key => $value) {
            $newKey = $prefix ? "{$prefix}.{$key}" : $key;
            
            if (is_array($value)) {
                $result = array_merge($result, $this->flattenArray($value, $newKey));
            } else {
                $result[] = $newKey;
            }
        }
        
        return $result;
    }
    
    private function fixMissingKeys($locale, $missingKeys)
    {
        $this->info("Fixing missing keys for {$locale}...");
        
        foreach ($missingKeys as $key) {
            $parts = explode('.', $key);
            
            if (count($parts) > 1) {
                // PHP array format
                $filename = array_shift($parts);
                $this->addToPhpFile($locale, $filename, $parts, $key);
            } else {
                // JSON format
                $this->addToJsonFile($locale, $key);
            }
        }
    }
    
    private function addToPhpFile($locale, $filename, $keyParts, $originalKey)
    {
        $filePath = resource_path("lang/{$locale}/{$filename}.php");
        
        if (!File::exists($filePath)) {
            $content = "<?php\n\nreturn [\n    // Auto-generated translations\n];\n";
            File::put($filePath, $content);
        }
        
        $translations = include $filePath;
        
        // Add the missing key with a placeholder
        $current = &$translations;
        foreach ($keyParts as $part) {
            if (!isset($current[$part])) {
                $current[$part] = [];
            }
            $current = &$current[$part];
        }
        
        // Set the final value
        $finalKey = array_pop($keyParts);
        $current[$finalKey] = "[TODO: Translate '{$originalKey}']";
        
        // Write back to file
        $content = "<?php\n\nreturn " . var_export($translations, true) . ";\n";
        File::put($filePath, $content);
        
        $this->line("  + Added {$originalKey} to {$filename}.php");
    }
    
    private function addToJsonFile($locale, $key)
    {
        $filePath = resource_path("lang/{$locale}.json");
        
        $translations = [];
        if (File::exists($filePath)) {
            $translations = json_decode(File::get($filePath), true) ?: [];
        }
        
        $translations[$key] = "[TODO: Translate '{$key}']";
        
        File::put($filePath, json_encode($translations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        $this->line("  + Added {$key} to {$locale}.json");
    }
}