<?php

namespace App\Helpers;

use Carbon\Carbon;
use Illuminate\Support\Facades\App;

class LocaleHelper
{
    /**
     * Format number according to current locale
     */
    public static function formatNumber($number, $decimals = 0): string
    {
        $locale = App::getLocale();
        
        if ($locale === 'th') {
            return number_format($number, $decimals, '.', ',');
        }
        
        return number_format($number, $decimals, '.', ',');
    }
    
    /**
     * Format currency according to current locale
     */
    public static function formatCurrency($amount, $decimals = 2): string
    {
        $locale = App::getLocale();
        $formatted = self::formatNumber($amount, $decimals);
        
        if ($locale === 'th') {
            return '฿ ' . $formatted;
        }
        
        return '$ ' . $formatted;
    }
    
    /**
     * Format date according to current locale
     */
    public static function formatDate($date, $format = null): string
    {
        if (!$date) return '';
        
        $carbon = $date instanceof Carbon ? $date : Carbon::parse($date);
        $locale = App::getLocale();
        
        if ($locale === 'th') {
            $carbon->locale('th');
            return $format ? $carbon->format($format) : $carbon->format('j F Y');
        }
        
        $carbon->locale('en');
        return $format ? $carbon->format($format) : $carbon->format('F j, Y');
    }
    
    /**
     * Format datetime according to current locale
     */
    public static function formatDateTime($datetime, $format = null): string
    {
        if (!$datetime) return '';
        
        $carbon = $datetime instanceof Carbon ? $datetime : Carbon::parse($datetime);
        $locale = App::getLocale();
        
        if ($locale === 'th') {
            $carbon->locale('th');
            return $format ? $carbon->format($format) : $carbon->format('j F Y เวลา H:i น.');
        }
        
        $carbon->locale('en');
        return $format ? $carbon->format($format) : $carbon->format('F j, Y at g:i A');
    }
    
    /**
     * Get month names for current locale
     */
    public static function getMonthNames(): array
    {
        $locale = App::getLocale();
        
        if ($locale === 'th') {
            return [
                1 => 'มกราคม', 2 => 'กุมภาพันธ์', 3 => 'มีนาคม', 4 => 'เมษายน',
                5 => 'พฤษภาคม', 6 => 'มิถุนายน', 7 => 'กรกฎาคม', 8 => 'สิงหาคม',
                9 => 'กันยายน', 10 => 'ตุลาคม', 11 => 'พฤศจิกายน', 12 => 'ธันวาคม'
            ];
        }
        
        return [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];
    }
    
    /**
     * Get day names for current locale
     */
    public static function getDayNames(): array
    {
        $locale = App::getLocale();
        
        if ($locale === 'th') {
            return [
                0 => 'อาทิตย์', 1 => 'จันทร์', 2 => 'อังคาร', 3 => 'พุธ',
                4 => 'พฤหัสบดี', 5 => 'ศุกร์', 6 => 'เสาร์'
            ];
        }
        
        return [
            0 => 'Sunday', 1 => 'Monday', 2 => 'Tuesday', 3 => 'Wednesday',
            4 => 'Thursday', 5 => 'Friday', 6 => 'Saturday'
        ];
    }
    
    /**
     * Get short day names for current locale
     */
    public static function getShortDayNames(): array
    {
        $locale = App::getLocale();
        
        if ($locale === 'th') {
            return [
                0 => 'อา.', 1 => 'จ.', 2 => 'อ.', 3 => 'พ.',
                4 => 'พฤ.', 5 => 'ศ.', 6 => 'ส.'
            ];
        }
        
        return [
            0 => 'Sun', 1 => 'Mon', 2 => 'Tue', 3 => 'Wed',
            4 => 'Thu', 5 => 'Fri', 6 => 'Sat'
        ];
    }
    
    /**
     * Format relative time (e.g., "2 hours ago")
     */
    public static function formatRelativeTime($datetime): string
    {
        if (!$datetime) return '';
        
        $carbon = $datetime instanceof Carbon ? $datetime : Carbon::parse($datetime);
        $locale = App::getLocale();
        
        $carbon->locale($locale);
        return $carbon->diffForHumans();
    }
    
    /**
     * Get current locale direction (ltr/rtl)
     */
    public static function getDirection(): string
    {
        return 'ltr'; // Both Thai and English are LTR
    }
    
    /**
     * Get current locale name
     */
    public static function getCurrentLocaleName(): string
    {
        $locale = App::getLocale();
        $names = config('app.locale_names', []);
        
        return $names[$locale] ?? $locale;
    }
}