<?php

namespace App\Http\Controllers;

use App\Models\FinanceRecord;
use App\Models\FinanceCategory;
use App\Models\MonthlyBudget;
use App\Models\SavingGoal;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class FinanceReportController extends Controller
{
    public function index()
    {
        try {
            $user = auth()->user();
            
            if (!$user) {
                return redirect()->route('login');
            }
            
            // Get available years
            $years = FinanceRecord::forUser($user->id)
                ->selectRaw('YEAR(record_date) as year')
                ->distinct()
                ->orderBy('year', 'desc')
                ->pluck('year');
            
            if ($years->isEmpty()) {
                $years = collect([now()->year]);
            }
            
            return view('finance.reports.index', compact('years'));
            
        } catch (\Exception $e) {
            return back()->with('error', 'เกิดข้อผิดพลาดในการโหลดรายงาน');
        }
    }

    public function monthly(Request $request)
    {
        $user = auth()->user();
        $month = $request->input('month', now()->format('Y-m'));
        
        // Get monthly data
        $monthlyStats = FinanceRecord::getMonthlyStats($user->id, $month);
        
        // Get daily breakdown
        $dailyData = FinanceRecord::getDailySummary($user->id, $month);
        
        // Get budget comparison
        $budgets = MonthlyBudget::forUser($user->id)
            ->forMonth($month)
            ->with('categoryInfo')
            ->get();
        
        foreach ($budgets as $budget) {
            $budget->updateSpentAmount();
        }
        
        return view('finance.reports.monthly', compact(
            'month',
            'monthlyStats',
            'dailyData',
            'budgets'
        ));
    }

    public function yearly(Request $request)
    {
        $user = auth()->user();
        $year = $request->input('year', now()->year);
        
        // Get yearly trend
        $yearlyTrend = FinanceRecord::getYearlyTrend($user->id, $year);
        
        // Calculate yearly totals
        $yearlyTotals = [
            'income' => collect($yearlyTrend)->sum('income'),
            'expense' => collect($yearlyTrend)->sum('expense'),
            'net' => collect($yearlyTrend)->sum('net'),
            'avg_monthly_income' => collect($yearlyTrend)->avg('income'),
            'avg_monthly_expense' => collect($yearlyTrend)->avg('expense'),
        ];
        
        // Get category breakdown for the year
        $categoryBreakdown = FinanceRecord::forUser($user->id)
            ->forYear($year)
            ->with('categoryInfo')
            ->get()
            ->groupBy(['type', 'category'])
            ->map(function ($typeGroup) {
                return $typeGroup->map(function ($records) {
                    return [
                        'amount' => $records->sum('amount'),
                        'count' => $records->count(),
                        'category_info' => $records->first()->categoryInfo,
                    ];
                })->sortByDesc('amount');
            });
        
        return view('finance.reports.yearly', compact(
            'year',
            'yearlyTrend',
            'yearlyTotals',
            'categoryBreakdown'
        ));
    }

    public function byCategory(Request $request)
    {
        $user = auth()->user();
        $period = $request->input('period', 'month');
        $date = $request->input('date', now()->format('Y-m'));
        
        $query = FinanceRecord::forUser($user->id)->with('categoryInfo');
        
        if ($period === 'month') {
            $query->forMonth($date);
        } elseif ($period === 'year') {
            $query->forYear($date);
        }
        
        // Get data grouped by category
        $categoryData = $query->get()
            ->groupBy(['type', 'category'])
            ->map(function ($typeGroup) {
                return $typeGroup->map(function ($records) {
                    $category = $records->first()->categoryInfo;
                    return [
                        'category' => $category ? $category->name : $records->first()->category,
                        'icon' => $category ? $category->icon : 'fa-circle',
                        'color' => $category ? $category->color : '#6B7280',
                        'amount' => $records->sum('amount'),
                        'count' => $records->count(),
                        'percentage' => 0, // Will calculate later
                        'records' => $records->sortByDesc('record_date')->take(5),
                    ];
                })->sortByDesc('amount');
            });
        
        // Calculate percentages
        if (isset($categoryData['income'])) {
            $totalIncome = $categoryData['income']->sum('amount');
            $categoryData['income'] = $categoryData['income']->map(function ($item) use ($totalIncome) {
                $item['percentage'] = $totalIncome > 0 ? round(($item['amount'] / $totalIncome) * 100, 1) : 0;
                return $item;
            });
        }
        
        if (isset($categoryData['expense'])) {
            $totalExpense = $categoryData['expense']->sum('amount');
            $categoryData['expense'] = $categoryData['expense']->map(function ($item) use ($totalExpense) {
                $item['percentage'] = $totalExpense > 0 ? round(($item['amount'] / $totalExpense) * 100, 1) : 0;
                return $item;
            });
        }
        
        return view('finance.reports.category', compact(
            'period',
            'date',
            'categoryData'
        ));
    }

    public function export(Request $request)
    {
        $user = auth()->user();
        $type = $request->input('type', 'monthly');
        $period = $request->input('period', now()->format('Y-m'));
        $format = $request->input('format', 'csv');
        
        switch ($type) {
            case 'monthly':
                return $this->exportMonthly($user->id, $period, $format);
            case 'yearly':
                return $this->exportYearly($user->id, $period, $format);
            case 'category':
                return $this->exportCategory($user->id, $period, $format);
            default:
                return back()->with('error', 'ประเภทรายงานไม่ถูกต้อง');
        }
    }

    private function exportMonthly($userId, $month, $format)
    {
        $records = FinanceRecord::forUser($userId)
            ->forMonth($month)
            ->with('categoryInfo')
            ->orderBy('record_date')
            ->orderBy('created_at')
            ->get();
        
        $stats = FinanceRecord::getMonthlyStats($userId, $month);
        
        if ($format === 'csv') {
            return $this->generateCSV($records, $stats, $month, 'monthly');
        }
        
        // For PDF, return view
        return view('finance.reports.export-pdf', compact('records', 'stats', 'month'));
    }

    private function exportYearly($userId, $year, $format)
    {
        $records = FinanceRecord::forUser($userId)
            ->forYear($year)
            ->with('categoryInfo')
            ->orderBy('record_date')
            ->orderBy('created_at')
            ->get();
        
        $yearlyTrend = FinanceRecord::getYearlyTrend($userId, $year);
        
        if ($format === 'csv') {
            return $this->generateYearlyCSV($records, $yearlyTrend, $year);
        }
        
        return view('finance.reports.export-yearly-pdf', compact('records', 'yearlyTrend', 'year'));
    }

    private function generateCSV($records, $stats, $period, $type)
    {
        $filename = "finance_report_{$type}_{$period}.csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($records, $stats, $period) {
            $file = fopen('php://output', 'w');
            
            // UTF-8 BOM
            fwrite($file, "\xEF\xBB\xBF");
            
            // Title
            fputcsv($file, ['รายงานการเงิน ประจำ ' . Carbon::parse($period)->locale('th')->format('F Y')]);
            fputcsv($file, []);
            
            // Summary
            fputcsv($file, ['สรุป']);
            fputcsv($file, ['รายการ', 'จำนวนเงิน (บาท)']);
            fputcsv($file, ['รายรับทั้งหมด', number_format($stats['total_income'], 2)]);
            fputcsv($file, ['รายจ่ายทั้งหมด', number_format($stats['total_expense'], 2)]);
            fputcsv($file, ['คงเหลือ', number_format($stats['net_amount'], 2)]);
            fputcsv($file, []);
            
            // Category breakdown
            fputcsv($file, ['สรุปตามหมวดหมู่']);
            fputcsv($file, ['หมวดหมู่', 'จำนวนเงิน', 'เปอร์เซ็นต์']);
            foreach ($stats['expense_by_category'] as $cat) {
                $category = FinanceCategory::where('slug', $cat['category'])->first();
                fputcsv($file, [
                    $category ? $category->name : $cat['category'],
                    number_format($cat['amount'], 2),
                    $cat['percentage'] . '%'
                ]);
            }
            fputcsv($file, []);
            
            // Detail records
            fputcsv($file, ['รายการทั้งหมด']);
            fputcsv($file, ['วันที่', 'ประเภท', 'หมวดหมู่', 'รายการ', 'จำนวนเงิน', 'แหล่งที่มา']);
            
            foreach ($records as $record) {
                fputcsv($file, [
                    $record->record_date->format('d/m/Y'),
                    $record->type === 'income' ? 'รายรับ' : 'รายจ่าย',
                    $record->categoryInfo ? $record->categoryInfo->name : $record->category,
                    $record->note ?? '-',
                    number_format($record->amount, 2),
                    $record->source === 'delivery' ? 'จากการจัดส่ง' : 'บันทึกเอง'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function generateYearlyCSV($records, $yearlyTrend, $year)
    {
        $filename = "finance_report_yearly_{$year}.csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($records, $yearlyTrend, $year) {
            $file = fopen('php://output', 'w');
            
            // UTF-8 BOM
            fwrite($file, "\xEF\xBB\xBF");
            
            // Title
            fputcsv($file, ['รายงานการเงินประจำปี ' . $year]);
            fputcsv($file, []);
            
            // Monthly summary
            fputcsv($file, ['สรุปรายเดือน']);
            fputcsv($file, ['เดือน', 'รายรับ', 'รายจ่าย', 'คงเหลือ']);
            
            foreach ($yearlyTrend as $month) {
                fputcsv($file, [
                    $month['month_name'],
                    number_format($month['income'], 2),
                    number_format($month['expense'], 2),
                    number_format($month['net'], 2)
                ]);
            }
            
            fputcsv($file, []);
            
            // Yearly totals
            $totalIncome = collect($yearlyTrend)->sum('income');
            $totalExpense = collect($yearlyTrend)->sum('expense');
            $totalNet = collect($yearlyTrend)->sum('net');
            
            fputcsv($file, ['สรุปทั้งปี']);
            fputcsv($file, ['รายรับรวม', number_format($totalIncome, 2)]);
            fputcsv($file, ['รายจ่ายรวม', number_format($totalExpense, 2)]);
            fputcsv($file, ['คงเหลือสุทธิ', number_format($totalNet, 2)]);
            fputcsv($file, ['รายรับเฉลี่ยต่อเดือน', number_format($totalIncome / 12, 2)]);
            fputcsv($file, ['รายจ่ายเฉลี่ยต่อเดือน', number_format($totalExpense / 12, 2)]);

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}