<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Cookie;

class LocaleMiddleware
{
    public function handle(Request $request, Closure $next)
    {
        $locale = $this->getLocale($request);
        
        if (in_array($locale, config('app.available_locales', ['th', 'en']))) {
            App::setLocale($locale);
            Session::put('locale', $locale);
        }
        
        return $next($request);
    }
    
    private function getLocale(Request $request): string
    {
        // 1. Query parameter ?lang=
        if ($request->has('lang') && in_array($request->get('lang'), config('app.available_locales'))) {
            $locale = $request->get('lang');
            Cookie::queue('locale', $locale, 525600); // 1 year
            return $locale;
        }
        
        // 2. Cookie/localStorage
        if ($cookieLocale = $request->cookie('locale')) {
            if (in_array($cookieLocale, config('app.available_locales'))) {
                return $cookieLocale;
            }
        }
        
        // 3. Session
        if ($sessionLocale = Session::get('locale')) {
            if (in_array($sessionLocale, config('app.available_locales'))) {
                return $sessionLocale;
            }
        }
        
        // 4. URL prefix (/th, /en) - for future implementation
        $segments = $request->segments();
        if (!empty($segments) && in_array($segments[0], config('app.available_locales'))) {
            return $segments[0];
        }
        
        // 5. Accept-Language header
        $acceptLanguage = $request->header('Accept-Language');
        if ($acceptLanguage) {
            $languages = explode(',', $acceptLanguage);
            foreach ($languages as $lang) {
                $lang = trim(explode(';', $lang)[0]);
                $lang = substr($lang, 0, 2); // Get first 2 chars
                if (in_array($lang, config('app.available_locales'))) {
                    return $lang;
                }
            }
        }
        
        // 6. Fallback to default
        return config('app.locale', 'th');
    }
}