/**
 * Responsive Utilities for EazyCom Plus
 * Modern JavaScript utilities for responsive behavior
 */

class ResponsiveUtils {
    constructor() {
        this.breakpoints = {
            sm: 640,
            md: 768,
            lg: 1024,
            xl: 1280,
            '2xl': 1536
        };
        
        this.init();
    }

    init() {
        this.setupViewportHandler();
        this.setupTouchHandlers();
        this.setupResizeObserver();
        this.setupIntersectionObserver();
    }

    // Viewport and breakpoint utilities
    getCurrentBreakpoint() {
        const width = window.innerWidth;
        
        if (width >= this.breakpoints['2xl']) return '2xl';
        if (width >= this.breakpoints.xl) return 'xl';
        if (width >= this.breakpoints.lg) return 'lg';
        if (width >= this.breakpoints.md) return 'md';
        if (width >= this.breakpoints.sm) return 'sm';
        return 'xs';
    }

    isMobile() {
        return window.innerWidth < this.breakpoints.md;
    }

    isTablet() {
        return window.innerWidth >= this.breakpoints.md && window.innerWidth < this.breakpoints.lg;
    }

    isDesktop() {
        return window.innerWidth >= this.breakpoints.lg;
    }

    // Responsive image loading
    setupResponsiveImages() {
        const images = document.querySelectorAll('img[data-src-mobile], img[data-src-tablet], img[data-src-desktop]');
        
        images.forEach(img => {
            this.updateImageSource(img);
        });
    }

    updateImageSource(img) {
        const breakpoint = this.getCurrentBreakpoint();
        let src = img.dataset.src;

        if (this.isMobile() && img.dataset.srcMobile) {
            src = img.dataset.srcMobile;
        } else if (this.isTablet() && img.dataset.srcTablet) {
            src = img.dataset.srcTablet;
        } else if (this.isDesktop() && img.dataset.srcDesktop) {
            src = img.dataset.srcDesktop;
        }

        if (src && img.src !== src) {
            img.src = src;
        }
    }

    // Touch and gesture handling
    setupTouchHandlers() {
        // Add touch feedback to buttons
        const touchElements = document.querySelectorAll('.btn, .nav-item, .quick-action, .card');
        
        touchElements.forEach(element => {
            element.addEventListener('touchstart', this.handleTouchStart.bind(this), { passive: true });
            element.addEventListener('touchend', this.handleTouchEnd.bind(this), { passive: true });
        });

        // Setup swipe gestures
        this.setupSwipeGestures();
    }

    handleTouchStart(event) {
        const element = event.currentTarget;
        element.style.transform = 'scale(0.98)';
        element.style.transition = 'transform 0.1s ease';
    }

    handleTouchEnd(event) {
        const element = event.currentTarget;
        setTimeout(() => {
            element.style.transform = '';
            element.style.transition = '';
        }, 100);
    }

    setupSwipeGestures() {
        let startX = 0;
        let startY = 0;
        let endX = 0;
        let endY = 0;

        document.addEventListener('touchstart', (e) => {
            startX = e.changedTouches[0].screenX;
            startY = e.changedTouches[0].screenY;
        }, { passive: true });

        document.addEventListener('touchend', (e) => {
            endX = e.changedTouches[0].screenX;
            endY = e.changedTouches[0].screenY;
            this.handleSwipe(startX, startY, endX, endY);
        }, { passive: true });
    }

    handleSwipe(startX, startY, endX, endY) {
        const deltaX = endX - startX;
        const deltaY = endY - startY;
        const minSwipeDistance = 50;

        if (Math.abs(deltaX) > Math.abs(deltaY) && Math.abs(deltaX) > minSwipeDistance) {
            if (deltaX > 0) {
                this.onSwipeRight();
            } else {
                this.onSwipeLeft();
            }
        }
    }

    onSwipeRight() {
        // Open sidebar on mobile
        if (this.isMobile()) {
            const sidebar = document.getElementById('slideMenu');
            if (sidebar && !sidebar.classList.contains('active')) {
                this.openSlideMenu();
            }
        }
    }

    onSwipeLeft() {
        // Close sidebar on mobile
        if (this.isMobile()) {
            const sidebar = document.getElementById('slideMenu');
            if (sidebar && sidebar.classList.contains('active')) {
                this.closeSlideMenu();
            }
        }
    }

    // Resize observer for dynamic layouts
    setupResizeObserver() {
        if ('ResizeObserver' in window) {
            const resizeObserver = new ResizeObserver(entries => {
                entries.forEach(entry => {
                    this.handleElementResize(entry);
                });
            });

            // Observe responsive containers
            const containers = document.querySelectorAll('.container, .stats-grid, .content-grid');
            containers.forEach(container => {
                resizeObserver.observe(container);
            });
        }
    }

    handleElementResize(entry) {
        const element = entry.target;
        const width = entry.contentRect.width;

        // Add responsive classes based on container width
        element.classList.remove('container-sm', 'container-md', 'container-lg', 'container-xl');
        
        if (width >= 1280) {
            element.classList.add('container-xl');
        } else if (width >= 1024) {
            element.classList.add('container-lg');
        } else if (width >= 768) {
            element.classList.add('container-md');
        } else if (width >= 640) {
            element.classList.add('container-sm');
        }
    }

    // Intersection observer for animations
    setupIntersectionObserver() {
        if ('IntersectionObserver' in window) {
            const observer = new IntersectionObserver(entries => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animate-in');
                    }
                });
            }, {
                threshold: 0.1,
                rootMargin: '0px 0px -50px 0px'
            });

            // Observe elements that should animate in
            const animateElements = document.querySelectorAll('.stat-card, .quick-action, .activity-item');
            animateElements.forEach(element => {
                observer.observe(element);
            });
        }
    }

    // Viewport height fix for mobile browsers
    setupViewportHandler() {
        const setVH = () => {
            const vh = window.innerHeight * 0.01;
            document.documentElement.style.setProperty('--vh', `${vh}px`);
        };

        setVH();
        window.addEventListener('resize', setVH);
        window.addEventListener('orientationchange', () => {
            setTimeout(setVH, 100);
        });
    }

    // Responsive font size calculator
    getResponsiveFontSize(baseSize, minSize = 12, maxSize = 24) {
        const viewportWidth = window.innerWidth;
        const minViewport = 320;
        const maxViewport = 1200;
        
        const ratio = (viewportWidth - minViewport) / (maxViewport - minViewport);
        const fontSize = baseSize + (ratio * (maxSize - baseSize));
        
        return Math.max(minSize, Math.min(maxSize, fontSize));
    }

    // Responsive spacing calculator
    getResponsiveSpacing(baseSpacing, multiplier = 1.5) {
        const breakpoint = this.getCurrentBreakpoint();
        const spacingMap = {
            xs: baseSpacing * 0.75,
            sm: baseSpacing,
            md: baseSpacing * 1.25,
            lg: baseSpacing * multiplier,
            xl: baseSpacing * (multiplier * 1.25),
            '2xl': baseSpacing * (multiplier * 1.5)
        };
        
        return spacingMap[breakpoint] || baseSpacing;
    }

    // Lazy loading for images
    setupLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove('lazy');
                        observer.unobserve(img);
                    }
                });
            });

            const lazyImages = document.querySelectorAll('img[data-src]');
            lazyImages.forEach(img => imageObserver.observe(img));
        }
    }

    // Performance monitoring
    measurePerformance() {
        if ('performance' in window) {
            const navigation = performance.getEntriesByType('navigation')[0];
            const paintEntries = performance.getEntriesByType('paint');
            
            console.log('Page Load Performance:', {
                domContentLoaded: navigation.domContentLoadedEventEnd - navigation.domContentLoadedEventStart,
                loadComplete: navigation.loadEventEnd - navigation.loadEventStart,
                firstPaint: paintEntries.find(entry => entry.name === 'first-paint')?.startTime,
                firstContentfulPaint: paintEntries.find(entry => entry.name === 'first-contentful-paint')?.startTime
            });
        }
    }

    // Utility methods for external use
    openSlideMenu() {
        const menu = document.getElementById('slideMenu');
        const overlay = document.getElementById('overlay');
        
        if (menu) menu.classList.add('active');
        if (overlay) overlay.classList.add('active');
        document.body.style.overflow = 'hidden';
    }

    closeSlideMenu() {
        const menu = document.getElementById('slideMenu');
        const overlay = document.getElementById('overlay');
        
        if (menu) menu.classList.remove('active');
        if (overlay) overlay.classList.remove('active');
        document.body.style.overflow = '';
    }

    // Debounce utility
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    // Throttle utility
    throttle(func, limit) {
        let inThrottle;
        return function() {
            const args = arguments;
            const context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }
}

// Initialize responsive utilities
const responsiveUtils = new ResponsiveUtils();

// Export for use in other scripts
window.ResponsiveUtils = ResponsiveUtils;
window.responsiveUtils = responsiveUtils;

// Add CSS for animations
const animationCSS = `
<style>
.animate-in {
    animation: slideInUp 0.6s ease-out forwards;
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.lazy {
    opacity: 0;
    transition: opacity 0.3s;
}

.lazy.loaded {
    opacity: 1;
}

/* CSS Custom Properties for responsive values */
:root {
    --responsive-padding: clamp(1rem, 4vw, 2rem);
    --responsive-margin: clamp(0.5rem, 2vw, 1rem);
    --responsive-font-sm: clamp(0.75rem, 2vw, 0.875rem);
    --responsive-font-base: clamp(0.875rem, 2.5vw, 1rem);
    --responsive-font-lg: clamp(1rem, 3vw, 1.25rem);
    --responsive-font-xl: clamp(1.25rem, 4vw, 2rem);
}
</style>
`;

// Inject animation CSS
document.head.insertAdjacentHTML('beforeend', animationCSS);