@extends('layouts.app')

@section('title', 'แก้ไขอัตราค่าคอมมิชชั่น - ระบบจัดการค่าคอมมิชชั่น')

@section('content')
<div class="edit-commission-container">
    <!-- Header Card -->
    <div class="header-card">
        <div class="header-content">
            <div class="header-info">
                <div class="header-icon">
                    <i class="fas fa-edit"></i>
                </div>
                <div class="header-text">
                    <h1 class="header-title">แก้ไขอัตราค่าคอมมิชชั่น</h1>
                    <p class="header-subtitle">{{ $commissionRate->warehouse_code }} - {{ $commissionRate->vehicle_type == '2wheel' ? 'รถ 2 ล้อ' : 'รถ 4 ล้อ' }} (ขั้นที่ {{ $commissionRate->tier }})</p>
                </div>
            </div>
            <div class="header-actions">
                <button onclick="deleteRate()" class="btn btn-danger">
                    <i class="fas fa-trash"></i>
                    <span>ลบ</span>
                </button>
                <a href="{{ route('admin.commission-rates.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i>
                    <span>กลับ</span>
                </a>
            </div>
        </div>
    </div>

    <form method="POST" action="{{ route('admin.commission-rates.update', $commissionRate) }}" id="commissionForm">
        @csrf
        @method('PUT')
        
        <div class="form-grid">
            <!-- Basic Information Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-icon basic">
                        <i class="fas fa-info-circle"></i>
                    </div>
                    <div class="card-title">ข้อมูลพื้นฐาน</div>
                </div>
                <div class="card-content">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="warehouse_code" class="form-label">คลังสินค้า <span class="required">*</span></label>
                            <select id="warehouse_code" name="warehouse_code" class="form-control @error('warehouse_code') error @enderror" required>
                                <option value="">เลือกคลังสินค้า</option>
                                @foreach($warehouses ?? [] as $warehouse)
                                    <option value="{{ $warehouse->code }}" {{ old('warehouse_code', $commissionRate->warehouse_code) === $warehouse->code ? 'selected' : '' }}>
                                        {{ $warehouse->code }} - {{ $warehouse->name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('warehouse_code')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="vehicle_type" class="form-label">ประเภทรถ <span class="required">*</span></label>
                            <select id="vehicle_type" name="vehicle_type" class="form-control @error('vehicle_type') error @enderror" required>
                                <option value="">เลือกประเภทรถ</option>
                                <option value="2wheel" {{ old('vehicle_type', $commissionRate->vehicle_type) === '2wheel' ? 'selected' : '' }}>รถ 2 ล้อ</option>
                                <option value="4wheel" {{ old('vehicle_type', $commissionRate->vehicle_type) === '4wheel' ? 'selected' : '' }}>รถ 4 ล้อ</option>
                            </select>
                            @error('vehicle_type')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="tier" class="form-label">ขั้น <span class="required">*</span></label>
                            <input type="number" id="tier" name="tier" value="{{ old('tier', $commissionRate->tier) }}" 
                                   class="form-control @error('tier') error @enderror" min="1" placeholder="เช่น 1, 2, 3" required>
                            @error('tier')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                            <div class="form-hint">ขั้นต้องไม่ซ้ำกันสำหรับคลังและประเภทรถเดียวกัน</div>
                        </div>

                        <div class="form-group">
                            <label class="form-label">สถานะ</label>
                            <div class="radio-group">
                                <label class="radio-item">
                                    <input type="radio" name="is_active" value="1" {{ old('is_active', $commissionRate->is_active) ? 'checked' : '' }}>
                                    <span class="radio-mark"></span>
                                    <span class="radio-text">ใช้งานอยู่</span>
                                </label>
                                <label class="radio-item">
                                    <input type="radio" name="is_active" value="0" {{ !old('is_active', $commissionRate->is_active) ? 'checked' : '' }}>
                                    <span class="radio-mark"></span>
                                    <span class="radio-text">ปิดใช้งาน</span>
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Package Range Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-icon range">
                        <i class="fas fa-boxes"></i>
                    </div>
                    <div class="card-title">ช่วงจำนวนพัสดุ</div>
                </div>
                <div class="card-content">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="min_packages" class="form-label">จำนวนพัสดุขั้นต่ำ <span class="required">*</span></label>
                            <input type="number" id="min_packages" name="min_packages" value="{{ old('min_packages', $commissionRate->min_packages) }}" 
                                   class="form-control @error('min_packages') error @enderror" min="0" placeholder="เช่น 1, 50, 100" required>
                            @error('min_packages')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="max_packages" class="form-label">จำนวนพัสดุสูงสุด</label>
                            <input type="number" id="max_packages" name="max_packages" value="{{ old('max_packages', $commissionRate->max_packages) }}" 
                                   class="form-control @error('max_packages') error @enderror" min="1" placeholder="เว้นว่างหากไม่จำกัด">
                            @error('max_packages')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                            <div class="form-hint">เว้นว่างไว้หากเป็นขั้นสุดท้าย (ไม่จำกัดจำนวนสูงสุด)</div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="rate_per_package" class="form-label">อัตราค่าคอมมิชชั่นต่อชิ้น (บาท) <span class="required">*</span></label>
                        <input type="number" id="rate_per_package" name="rate_per_package" value="{{ old('rate_per_package', $commissionRate->rate_per_package) }}" 
                               class="form-control @error('rate_per_package') error @enderror" step="0.01" min="0" placeholder="เช่น 0, 8, 10" required>
                        @error('rate_per_package')
                            <div class="error-message">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- Calculation Preview Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-icon preview">
                        <i class="fas fa-calculator"></i>
                    </div>
                    <div class="card-title">ตัวอย่างการคำนวณ</div>
                </div>
                <div class="card-content">
                    <div id="calculation-preview" class="calculation-display">
                        <div class="calculation-placeholder">กรอกข้อมูลเพื่อดูตัวอย่างการคำนวณ</div>
                    </div>
                </div>
            </div>

            <!-- Existing Rates Card -->
            <div class="form-card" id="existing-rates-card" style="display: none;">
                <div class="card-header">
                    <div class="card-icon existing">
                        <i class="fas fa-list"></i>
                    </div>
                    <div class="card-title">อัตราอื่นๆ ที่มีอยู่</div>
                </div>
                <div class="card-content">
                    <div id="existing-rates-content" class="existing-rates-list">
                        <!-- จะแสดงผ่าน JavaScript -->
                    </div>
                </div>
            </div>

            <!-- Information Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-icon info">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="card-title">ข้อมูลเพิ่มเติม</div>
                </div>
                <div class="card-content">
                    <div class="form-row">
                        <div class="info-item">
                            <div class="info-label">สร้างเมื่อ</div>
                            <div class="info-value">{{ $commissionRate->created_at ? $commissionRate->created_at->format('d/m/Y H:i:s') : '-' }}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">อัปเดตล่าสุด</div>
                            <div class="info-value">{{ $commissionRate->updated_at ? $commissionRate->updated_at->format('d/m/Y H:i:s') : '-' }}</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Submit Actions -->
        <div class="submit-actions">
            <a href="{{ route('admin.commission-rates.index') }}" class="btn btn-secondary">
                <i class="fas fa-times"></i>
                <span>ยกเลิก</span>
            </a>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save"></i>
                <span>บันทึกการเปลี่ยนแปลง</span>
            </button>
        </div>
    </form>
</div>
@endsection

@push('styles')
<style>
/* Container */
.edit-commission-container {
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 50%, #cbd5e1 100%);
    padding: 2rem;
}

/* Header Card */
.header-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 24px;
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 2rem;
    margin-bottom: 2rem;
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1.5rem;
}

.header-info {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.header-icon {
    width: 64px;
    height: 64px;
    background: linear-gradient(135deg, #FF6B35 0%, #DC2626 100%);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: white;
    box-shadow: 0 10px 25px rgba(255, 107, 53, 0.3);
}

.header-title {
    font-size: 2rem;
    font-weight: 700;
    background: linear-gradient(135deg, #1f2937 0%, #6b7280 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    margin: 0 0 0.5rem 0;
}

.header-subtitle {
    color: #6b7280;
    margin: 0;
    font-size: 1rem;
}

.header-actions {
    display: flex;
    gap: 0.75rem;
}

/* Buttons */
.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    font-weight: 500;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #FF6B35 0%, #DC2626 100%);
    color: white;
    box-shadow: 0 10px 25px rgba(255, 107, 53, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 15px 35px rgba(255, 107, 53, 0.4);
}

.btn-secondary {
    background: rgba(255, 255, 255, 0.9);
    color: #374151;
    border: 1px solid #e5e7eb;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
}

.btn-secondary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
}

.btn-danger {
    background: linear-gradient(135deg, #EF4444 0%, #DC2626 100%);
    color: white;
    box-shadow: 0 10px 25px rgba(239, 68, 68, 0.3);
}

.btn-danger:hover {
    transform: translateY(-2px);
    box-shadow: 0 15px 35px rgba(239, 68, 68, 0.4);
}

/* Form Grid */
.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 2rem;
    margin-bottom: 2rem;
}

/* Form Cards */
.form-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    box-shadow: 0 20px 40px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    overflow: hidden;
    transition: all 0.3s ease;
}

.form-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.15);
}

.card-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1.5rem 2rem;
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-bottom: 1px solid #e5e7eb;
}

.card-icon {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.1rem;
}

.card-icon.basic {
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
}

.card-icon.range {
    background: linear-gradient(135deg, #10B981 0%, #059669 100%);
}

.card-icon.preview {
    background: linear-gradient(135deg, #F59E0B 0%, #D97706 100%);
}

.card-icon.existing {
    background: linear-gradient(135deg, #8B5CF6 0%, #7C3AED 100%);
}

.card-icon.info {
    background: linear-gradient(135deg, #6B7280 0%, #4B5563 100%);
}

.card-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: #1f2937;
}

.card-content {
    padding: 2rem;
}

/* Form Elements */
.form-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 1.5rem;
}

.form-row:last-child {
    margin-bottom: 0;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-label {
    font-weight: 500;
    color: #374151;
    font-size: 0.875rem;
}

.required {
    color: #EF4444;
}

.form-control {
    padding: 0.75rem;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: #FF6B35;
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.form-control.error {
    border-color: #EF4444;
}

.error-message {
    color: #EF4444;
    font-size: 0.75rem;
    margin-top: 0.25rem;
}

.form-hint {
    color: #6b7280;
    font-size: 0.75rem;
    margin-top: 0.25rem;
}

/* Radio Group */
.radio-group {
    display: flex;
    gap: 1.5rem;
}

.radio-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
}

.radio-item input[type="radio"] {
    display: none;
}

.radio-mark {
    width: 20px;
    height: 20px;
    border: 2px solid #e5e7eb;
    border-radius: 50%;
    position: relative;
    transition: all 0.3s ease;
}

.radio-item input[type="radio"]:checked + .radio-mark {
    border-color: #FF6B35;
    background: #FF6B35;
}

.radio-item input[type="radio"]:checked + .radio-mark::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 8px;
    height: 8px;
    background: white;
    border-radius: 50%;
}

.radio-text {
    font-size: 0.875rem;
    color: #374151;
}

/* Calculation Display */
.calculation-display {
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 2px solid #bae6fd;
    border-radius: 12px;
    padding: 1.5rem;
    min-height: 120px;
    display: flex;
    align-items: center;
}

.calculation-placeholder {
    color: #6b7280;
    font-style: italic;
    text-align: center;
    width: 100%;
}

.calculation-content {
    width: 100%;
}

.calculation-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #bae6fd;
}

.calculation-item:last-child {
    border-bottom: none;
    font-weight: 600;
    color: #FF6B35;
}

.calculation-label {
    color: #374151;
    font-size: 0.875rem;
}

.calculation-value {
    font-weight: 500;
    color: #1f2937;
}

/* Existing Rates */
.existing-rates-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.existing-rate-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 8px;
    border: 1px solid #e5e7eb;
}

.rate-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.rate-tier {
    font-weight: 600;
    color: #1f2937;
}

.rate-details {
    font-size: 0.875rem;
    color: #6b7280;
}

.rate-status {
    font-size: 0.75rem;
    font-weight: 500;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
}

.rate-status.active {
    background: #d1fae5;
    color: #047857;
}

.rate-status.inactive {
    background: #fee2e2;
    color: #dc2626;
}

/* Info Items */
.info-item {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.info-label {
    font-size: 0.875rem;
    color: #6b7280;
    font-weight: 500;
}

.info-value {
    padding: 0.75rem;
    background: #f8f9fa;
    border-radius: 8px;
    color: #1f2937;
    font-weight: 500;
}

/* Submit Actions */
.submit-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    box-shadow: 0 20px 40px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 2rem;
    gap: 1rem;
}

/* Responsive */
@media (max-width: 768px) {
    .edit-commission-container {
        padding: 1rem;
    }
    
    .header-content {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .header-actions {
        width: 100%;
        justify-content: stretch;
    }
    
    .btn {
        flex: 1;
        justify-content: center;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .radio-group {
        flex-direction: column;
        gap: 1rem;
    }
    
    .submit-actions {
        flex-direction: column-reverse;
        gap: 1rem;
    }
    
    .submit-actions .btn {
        width: 100%;
    }
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
// Real-time calculation preview
function updateCalculationPreview() {
    const minPackages = parseInt(document.getElementById('min_packages').value) || 0;
    const maxPackages = parseInt(document.getElementById('max_packages').value) || null;
    const rate = parseFloat(document.getElementById('rate_per_package').value) || 0;
    
    if (minPackages > 0 && rate >= 0) {
        const rangeText = maxPackages ? `${minPackages} - ${maxPackages}` : `${minPackages}+`;
        const exampleCount = maxPackages ? Math.min(minPackages + 10, maxPackages) : minPackages + 10;
        const calculatedPackages = exampleCount >= minPackages ? exampleCount - minPackages + 1 : 0;
        const exampleCommission = calculatedPackages * rate;
        
        document.getElementById('calculation-preview').innerHTML = `
            <div class="calculation-content">
                <div class="calculation-item">
                    <span class="calculation-label">ช่วงพัสดุ:</span>
                    <span class="calculation-value">${rangeText} ชิ้น</span>
                </div>
                <div class="calculation-item">
                    <span class="calculation-label">อัตราต่อชิ้น:</span>
                    <span class="calculation-value">${rate.toFixed(2)} บาท</span>
                </div>
                <div class="calculation-item">
                    <span class="calculation-label">ตัวอย่าง: พัสดุ ${exampleCount} ชิ้น</span>
                    <span class="calculation-value">${exampleCommission.toFixed(2)} บาท</span>
                </div>
            </div>
        `;
    } else {
        document.getElementById('calculation-preview').innerHTML = '<div class="calculation-placeholder">กรอกข้อมูลเพื่อดูตัวอย่างการคำนวณ</div>';
    }
}

// Load existing rates
function loadExistingRates() {
    const warehouseCode = document.getElementById('warehouse_code').value;
    const vehicleType = document.getElementById('vehicle_type').value;
    
    if (warehouseCode && vehicleType) {
        fetch(`{{ route('admin.commission-rates.existing') }}?warehouse_code=${warehouseCode}&vehicle_type=${vehicleType}`)
            .then(response => response.json())
            .then(data => {
                if (data.success && data.rates.length > 0) {
                    let html = '';
                    data.rates.forEach(rate => {
                        if (rate.id === {{ $commissionRate->id }}) return;
                        
                        const rangeText = rate.max_packages ? `${rate.min_packages} - ${rate.max_packages}` : `${rate.min_packages}+`;
                        const statusClass = rate.is_active ? 'active' : 'inactive';
                        const statusText = rate.is_active ? 'ใช้งานอยู่' : 'ปิดใช้งาน';
                        
                        html += `
                            <div class="existing-rate-item">
                                <div class="rate-info">
                                    <div class="rate-tier">ขั้นที่ ${rate.tier}</div>
                                    <div class="rate-details">${rangeText} ชิ้น = ${rate.rate_per_package} บาท/ชิ้น</div>
                                </div>
                                <div class="rate-status ${statusClass}">${statusText}</div>
                            </div>
                        `;
                    });
                    
                    if (html) {
                        document.getElementById('existing-rates-content').innerHTML = html;
                        document.getElementById('existing-rates-card').style.display = 'block';
                    } else {
                        document.getElementById('existing-rates-card').style.display = 'none';
                    }
                } else {
                    document.getElementById('existing-rates-card').style.display = 'none';
                }
            })
            .catch(error => {
                console.error('Error loading existing rates:', error);
                document.getElementById('existing-rates-card').style.display = 'none';
            });
    } else {
        document.getElementById('existing-rates-card').style.display = 'none';
    }
}

// Event listeners
document.getElementById('min_packages').addEventListener('input', updateCalculationPreview);
document.getElementById('max_packages').addEventListener('input', updateCalculationPreview);
document.getElementById('rate_per_package').addEventListener('input', updateCalculationPreview);
document.getElementById('warehouse_code').addEventListener('change', loadExistingRates);
document.getElementById('vehicle_type').addEventListener('change', loadExistingRates);

// Form validation
document.getElementById('commissionForm').addEventListener('submit', function(e) {
    const minPackages = parseInt(document.getElementById('min_packages').value);
    const maxPackages = parseInt(document.getElementById('max_packages').value);
    
    if (maxPackages && maxPackages <= minPackages) {
        e.preventDefault();
        Swal.fire({
            title: 'ข้อผิดพลาด',
            text: 'จำนวนพัสดุสูงสุดต้องมากกว่าจำนวนขั้นต่ำ',
            icon: 'error',
            confirmButtonColor: '#FF6B35'
        });
        return false;
    }
});

// Delete rate function
function deleteRate() {
    Swal.fire({
        title: 'ยืนยันการลบ',
        text: 'คุณต้องการลบอัตราค่าคอมมิชชั่นนี้หรือไม่? การดำเนินการนี้ไม่สามารถยกเลิกได้',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#EF4444',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'ลบ',
        cancelButtonText: 'ยกเลิก'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch(`{{ route('admin.commission-rates.destroy', $commissionRate) }}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    Swal.fire({
                        title: 'สำเร็จ!',
                        text: data.message,
                        icon: 'success',
                        confirmButtonColor: '#FF6B35'
                    }).then(() => {
                        window.location.href = '{{ route('admin.commission-rates.index') }}';
                    });
                } else {
                    Swal.fire({
                        title: 'เกิดข้อผิดพลาด!',
                        text: data.message,
                        icon: 'error',
                        confirmButtonColor: '#FF6B35'
                    });
                }
            })
            .catch(error => {
                Swal.fire({
                    title: 'เกิดข้อผิดพลาด!',
                    text: 'ไม่สามารถลบอัตราได้',
                    icon: 'error',
                    confirmButtonColor: '#FF6B35'
                });
            });
        }
    });
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    loadExistingRates();
    updateCalculationPreview();
});
</script>
@endpush