@extends('layouts.app')

@section('title', 'จัดการอัตราค่าคอมมิชชั่น - EazyCom Plus')
@section('page-title', 'จัดการอัตราค่าคอมมิชชั่น')
@section('mobile-title', 'อัตราคอมมิชชั่น')

@section('content')
<div class="commission-rates-page">
    <!-- Header Section -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-info">
                <div class="header-title">
                    <div class="title-icon">
                        <i class="fas fa-percentage"></i>
                    </div>
                    <div class="title-text">
                        <h1>จัดการอัตราค่าคอมมิชชั่น</h1>
                        <p>กำหนดและจัดการอัตราค่าคอมมิชชั่นสำหรับแต่ละคลังและประเภทรถ</p>
                    </div>
                </div>
                <div class="header-actions">
                    <button onclick="showCalculator()" class="action-btn info-btn">
                        <i class="fas fa-calculator"></i>
                        <span>คำนวณ</span>
                    </button>
                    <button onclick="exportRates()" class="action-btn success-btn">
                        <i class="fas fa-download"></i>
                        <span>Export</span>
                    </button>
                    <a href="{{ route('admin.commission-rates.create') }}" class="action-btn primary-btn">
                        <i class="fas fa-plus"></i>
                        <span>เพิ่มอัตราใหม่</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="stats-grid">
        <div class="stat-card total">
            <div class="stat-icon">
                <i class="fas fa-percentage"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ number_format($stats['total'] ?? 0) }}</div>
                <div class="stat-label">อัตราทั้งหมด</div>
                <div class="stat-trend">
                    <i class="fas fa-chart-line"></i>
                    <span>ทั้งระบบ</span>
                </div>
            </div>
            <div class="stat-action">
                <button onclick="clearFilters()" class="stat-link">ดูทั้งหมด</button>
            </div>
        </div>

        <div class="stat-card active">
            <div class="stat-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ number_format($stats['active'] ?? 0) }}</div>
                <div class="stat-label">อัตราที่ใช้งาน</div>
                <div class="stat-trend">
                    <i class="fas fa-check"></i>
                    <span>Active</span>
                </div>
            </div>
            <div class="stat-action">
                <button onclick="filterByStatus('active')" class="stat-link">กรองดู</button>
            </div>
        </div>

        <div class="stat-card warehouses">
            <div class="stat-icon">
                <i class="fas fa-warehouse"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ number_format($stats['warehouses'] ?? 0) }}</div>
                <div class="stat-label">คลังที่มีอัตรา</div>
                <div class="stat-trend">
                    <i class="fas fa-building"></i>
                    <span>Warehouses</span>
                </div>
            </div>
            <div class="stat-action">
                <button onclick="showWarehouses()" class="stat-link">จัดการคลัง</button>
            </div>
        </div>

        <div class="stat-card tiers">
            <div class="stat-icon">
                <i class="fas fa-layer-group"></i>
            </div>
            <div class="stat-content">
                <div class="stat-number">{{ number_format($stats['avg_tiers'] ?? 0, 1) }}</div>
                <div class="stat-label">ขั้นเฉลี่ย/คลัง</div>
                <div class="stat-trend">
                    <i class="fas fa-stairs"></i>
                    <span>Tiers</span>
                </div>
            </div>
            <div class="stat-action">
                <button onclick="showAnalytics()" class="stat-link">วิเคราะห์</button>
            </div>
        </div>
    </div>

    <!-- Filter Section -->
    <div class="filter-section">
        <div class="filter-header">
            <div class="filter-title">
                <i class="fas fa-filter"></i>
                <span>ตัวกรองและค้นหา</span>
            </div>
            <button class="filter-toggle" onclick="toggleFilters()">
                <i class="fas fa-chevron-down"></i>
            </button>
        </div>
        <div class="filter-content" id="filterContent">
            <form method="GET" action="{{ route('admin.commission-rates.index') }}" class="filter-form">
                <div class="filter-row">
                    <div class="filter-group">
                        <label class="filter-label">คลังสินค้า</label>
                        <div class="input-group">
                            <div class="input-icon">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <select name="warehouse" class="filter-select" onchange="this.form.submit()">
                                <option value="">ทั้งหมด</option>
                                @foreach($warehouses ?? [] as $warehouse)
                                    <option value="{{ $warehouse->code }}" 
                                            {{ request('warehouse') === $warehouse->code ? 'selected' : '' }}>
                                        {{ $warehouse->code }} - {{ $warehouse->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-group">
                        <label class="filter-label">ประเภทรถ</label>
                        <div class="input-group">
                            <div class="input-icon">
                                <i class="fas fa-truck"></i>
                            </div>
                            <select name="vehicle_type" class="filter-select" onchange="this.form.submit()">
                                <option value="">ทั้งหมด</option>
                                <option value="2wheel" {{ request('vehicle_type') === '2wheel' ? 'selected' : '' }}>
                                    🏍️ รถ 2 ล้อ
                                </option>
                                <option value="4wheel" {{ request('vehicle_type') === '4wheel' ? 'selected' : '' }}>
                                    🚚 รถ 4 ล้อ
                                </option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-group">
                        <label class="filter-label">สถานะ</label>
                        <div class="input-group">
                            <div class="input-icon">
                                <i class="fas fa-toggle-on"></i>
                            </div>
                            <select name="status" class="filter-select" onchange="this.form.submit()">
                                <option value="">ทั้งหมด</option>
                                <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>
                                    ✅ ใช้งานอยู่
                                </option>
                                <option value="inactive" {{ request('status') === 'inactive' ? 'selected' : '' }}>
                                    ❌ ปิดใช้งาน
                                </option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="filter-actions">
                        <a href="{{ route('admin.commission-rates.index') }}" class="filter-btn clear-btn">
                            <i class="fas fa-times"></i>
                            <span>ล้าง</span>
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Commission Rates Display -->
    <div class="rates-container">
        @if(isset($commissionRates) && $commissionRates->isNotEmpty())
            @php
                $groupedRates = $commissionRates->groupBy(['warehouse_code', 'vehicle_type']);
            @endphp

            @foreach($groupedRates as $warehouseCode => $warehouseRates)
                @php
                    $warehouse = $warehouses->firstWhere('code', $warehouseCode);
                @endphp
                
                <div class="warehouse-card">
                    <div class="warehouse-header">
                        <div class="warehouse-info">
                            <div class="warehouse-title">
                                <i class="fas fa-warehouse"></i>
                                <span>คลัง: {{ $warehouseCode }} - {{ $warehouse->name ?? 'ไม่พบชื่อคลัง' }}</span>
                            </div>
                            @if($warehouse && $warehouse->location)
                                <div class="warehouse-location">
                                    <i class="fas fa-map-marker-alt"></i>
                                    <span>{{ $warehouse->location }}</span>
                                </div>
                            @endif
                        </div>
                        <div class="warehouse-actions">
                            <button onclick="addRateForWarehouse('{{ $warehouseCode }}')" class="warehouse-btn primary">
                                <i class="fas fa-plus"></i>
                                <span>เพิ่มอัตรา</span>
                            </button>
                            <button onclick="bulkAction('{{ $warehouseCode }}')" class="warehouse-btn warning">
                                <i class="fas fa-cogs"></i>
                                <span>จัดการทั้งหมด</span>
                            </button>
                        </div>
                    </div>
                    
                    <div class="warehouse-content">
                        @foreach($warehouseRates as $vehicleType => $rates)
                            <div class="vehicle-section">
                                <div class="vehicle-header">
                                    <div class="vehicle-title">
                                        <i class="fas fa-{{ $vehicleType === '2wheel' ? 'motorcycle' : 'truck' }}"></i>
                                        <span>{{ $vehicleType === '2wheel' ? 'รถ 2 ล้อ' : 'รถ 4 ล้อ' }}</span>
                                        <div class="tier-count">{{ $rates->count() }} ขั้น</div>
                                    </div>
                                    <button onclick="previewCommission('{{ $warehouseCode }}', '{{ $vehicleType }}')" 
                                            class="vehicle-btn">
                                        <i class="fas fa-calculator"></i>
                                        <span>ทดสอบคำนวณ</span>
                                    </button>
                                </div>
                                
                                <div class="rates-list">
                                    @foreach($rates->sortBy('tier') as $rate)
                                        <div class="rate-item" data-rate-id="{{ $rate->id }}">
                                            <div class="rate-info">
                                                <div class="rate-tier">ขั้นที่ {{ $rate->tier }}</div>
                                                <div class="rate-range">
                                                    {{ number_format($rate->min_packages) }}
                                                    @if($rate->max_packages)
                                                        - {{ number_format($rate->max_packages) }}
                                                    @else
                                                        +
                                                    @endif
                                                    ชิ้น
                                                </div>
                                                <div class="rate-value">
                                                    {{ number_format($rate->rate_per_package, 2) }} บาท/ชิ้น
                                                </div>
                                            </div>
                                            
                                            <div class="rate-status">
                                                @if($rate->is_active)
                                                    <div class="status-badge active">
                                                        <i class="fas fa-check-circle"></i>
                                                        <span>ใช้งาน</span>
                                                    </div>
                                                @else
                                                    <div class="status-badge inactive">
                                                        <i class="fas fa-times-circle"></i>
                                                        <span>ปิดใช้งาน</span>
                                                    </div>
                                                @endif
                                            </div>
                                            
                                            <div class="rate-actions">
                                                <button onclick="editRate({{ $rate->id }})" 
                                                        class="rate-btn edit" title="แก้ไข">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button onclick="toggleRateStatus({{ $rate->id }})" 
                                                        class="rate-btn toggle" title="เปลี่ยนสถานะ">
                                                    <i class="fas fa-{{ $rate->is_active ? 'pause' : 'play' }}"></i>
                                                </button>
                                                <button onclick="deleteRate({{ $rate->id }})" 
                                                        class="rate-btn delete" title="ลบ">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </div>
                                        </div>
                                    @endforeach
                                </div>
                                
                                <!-- Commission Preview -->
                                <div class="commission-preview">
                                    <div class="preview-header">
                                        <i class="fas fa-info-circle"></i>
                                        <span>ตัวอย่าง: จัดส่ง 150 ชิ้น</span>
                                    </div>
                                    <div id="preview-{{ $warehouseCode }}-{{ $vehicleType }}" class="preview-result">
                                        <span class="loading-text">กำลังคำนวณ...</span>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            @endforeach
        @else
            <!-- Empty State -->
            <div class="empty-state">
                <div class="empty-content">
                    <div class="empty-icon">
                        <i class="fas fa-percentage"></i>
                    </div>
                    <div class="empty-text">
                        <h3>ยังไม่มีอัตราค่าคอมมิชชั่น</h3>
                        <p>เริ่มต้นการตั้งค่าอัตราค่าคอมมิชชั่นสำหรับคลังและประเภทรถต่างๆ</p>
                    </div>
                    <div class="empty-action">
                        <a href="{{ route('admin.commission-rates.create') }}" class="empty-btn">
                            <i class="fas fa-plus"></i>
                            <span>เพิ่มอัตราแรก</span>
                        </a>
                    </div>
                </div>
            </div>
        @endif
    </div>
</div>

<!-- Calculator Modal -->
<div id="calculatorModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title">
                <i class="fas fa-calculator"></i>
                คำนวณค่าคอมมิชชั่น
            </h3>
            <button onclick="closeCalculator()" class="modal-close">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body">
            <form id="calculatorForm" class="calculator-form">
                <div class="form-grid">
                    <div class="form-group">
                        <label class="form-label">คลังสินค้า</label>
                        <select id="calc_warehouse" name="warehouse_code" class="form-input" required>
                            <option value="">เลือกคลัง</option>
                            @foreach($warehouses ?? [] as $warehouse)
                                <option value="{{ $warehouse->code }}">
                                    {{ $warehouse->code }} - {{ $warehouse->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">ประเภทรถ</label>
                        <select id="calc_vehicle" name="vehicle_type" class="form-input" required>
                            <option value="">เลือกประเภท</option>
                            <option value="2wheel">🏍️ รถ 2 ล้อ</option>
                            <option value="4wheel">🚚 รถ 4 ล้อ</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">จำนวนพัสดุ</label>
                        <input type="number" id="calc_packages" name="package_count" 
                               class="form-input" min="1" placeholder="กรอกจำนวน" required>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-primary full-width">
                    <i class="fas fa-calculator"></i>
                    <span>คำนวณ</span>
                </button>
            </form>
            
            <div id="calculatorResult" class="calculator-result" style="display: none;">
                <!-- Results will be displayed here -->
            </div>
        </div>
    </div>
</div>

<style>
/* Commission Rates Page Styles */
.commission-rates-page {
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

/* Page Header */
.page-header {
    background: white;
    border-bottom: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    margin-bottom: 2rem;
}

.header-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 2rem;
}

.header-info {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 2rem;
}

.header-title {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.title-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 24px;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.3);
}

.title-text h1 {
    font-size: 2rem;
    font-weight: 600;
    color: #1e293b;
    margin: 0;
}

.title-text p {
    color: #64748b;
    margin: 0.5rem 0 0;
    font-size: 1rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
}

.action-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    text-decoration: none;
    font-weight: 500;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.primary-btn {
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    color: white;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.3);
}

.primary-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 53, 0.4);
}

.info-btn {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: white;
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
}

.info-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
}

.success-btn {
    background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
    color: white;
    box-shadow: 0 4px 12px rgba(34, 197, 94, 0.3);
}

.success-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(34, 197, 94, 0.4);
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    max-width: 1200px;
    margin: 0 auto 2rem;
    padding: 0 2rem;
}

.stat-card {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
    border: 1px solid #e2e8f0;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 30px rgba(0, 0, 0, 0.1);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
}

.stat-card.active::before {
    background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
}

.stat-card.warehouses::before {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
}

.stat-card.tiers::before {
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    color: white;
    margin-bottom: 1rem;
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
}

.stat-card.active .stat-icon {
    background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
}

.stat-card.warehouses .stat-icon {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
}

.stat-card.tiers .stat-icon {
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
}

.stat-number {
    font-size: 2.5rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 1rem;
    color: #64748b;
    margin-bottom: 1rem;
}

.stat-trend {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #64748b;
    font-size: 0.9rem;
    margin-bottom: 1rem;
}

.stat-action {
    margin-top: auto;
}

.stat-link {
    background: none;
    border: none;
    color: #FF6B35;
    font-weight: 500;
    cursor: pointer;
    text-decoration: underline;
    transition: color 0.3s ease;
}

.stat-link:hover {
    color: #e85d2f;
}

/* Filter Section */
.filter-section {
    max-width: 1200px;
    margin: 0 auto 2rem;
    padding: 0 2rem;
}

.filter-header {
    background: white;
    border-radius: 16px 16px 0 0;
    padding: 1.5rem 2rem;
    border: 1px solid #e2e8f0;
    border-bottom: none;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.filter-title {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 1.125rem;
    font-weight: 600;
    color: #1e293b;
}

.filter-title i {
    color: #FF6B35;
}

.filter-toggle {
    background: none;
    border: none;
    color: #64748b;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.filter-toggle:hover {
    background: #f1f5f9;
    color: #FF6B35;
}

.filter-content {
    background: white;
    border-radius: 0 0 16px 16px;
    border: 1px solid #e2e8f0;
    border-top: none;
    padding: 2rem;
}

.filter-row {
    display: grid;
    grid-template-columns: 1fr 1fr 1fr auto;
    gap: 1.5rem;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.filter-label {
    font-weight: 500;
    color: #374151;
    font-size: 0.95rem;
}

.input-group {
    position: relative;
}

.input-icon {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: #9ca3af;
    font-size: 0.9rem;
    z-index: 10;
}

.filter-select {
    width: 100%;
    padding: 0.875rem 1rem 0.875rem 2.75rem;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
    font-size: 0.95rem;
    transition: all 0.3s ease;
    background: white;
    font-family: inherit;
}

.filter-select:focus {
    outline: none;
    border-color: #FF6B35;
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.filter-actions {
    display: flex;
    gap: 0.5rem;
}

.filter-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.875rem 1.5rem;
    border-radius: 12px;
    text-decoration: none;
    font-weight: 500;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.clear-btn {
    background: #f1f5f9;
    color: #475569;
    border: 1px solid #e2e8f0;
}

.clear-btn:hover {
    background: #e2e8f0;
    transform: translateY(-1px);
}

/* Rates Container */
.rates-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 2rem 2rem;
}

/* Warehouse Card */
.warehouse-card {
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
    border: 1px solid #e2e8f0;
    margin-bottom: 2rem;
    overflow: hidden;
    transition: all 0.3s ease;
}

.warehouse-card:hover {
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
}

.warehouse-header {
    background: linear-gradient(135deg, rgba(255, 107, 53, 0.05) 0%, rgba(255, 140, 97, 0.05) 100%);
    padding: 1.5rem 2rem;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.warehouse-title {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 1.25rem;
    font-weight: 600;
    color: #1e293b;
}

.warehouse-title i {
    color: #FF6B35;
}

.warehouse-location {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #64748b;
    font-size: 0.9rem;
    margin-top: 0.5rem;
}

.warehouse-actions {
    display: flex;
    gap: 1rem;
}

.warehouse-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.25rem;
    border-radius: 12px;
    border: none;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.9rem;
}

.warehouse-btn.primary {
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    color: white;
    box-shadow: 0 2px 8px rgba(255, 107, 53, 0.3);
}

.warehouse-btn.primary:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.4);
}

.warehouse-btn.warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
    box-shadow: 0 2px 8px rgba(245, 158, 11, 0.3);
}

.warehouse-btn.warning:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(245, 158, 11, 0.4);
}

.warehouse-content {
    padding: 2rem;
}

/* Vehicle Section */
.vehicle-section {
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    background: rgba(255, 107, 53, 0.01);
}

.vehicle-section:last-child {
    margin-bottom: 0;
}

.vehicle-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 1.5rem;
}

.vehicle-title {
    display: flex;
    align-items: center;
    gap: 1rem;
    font-size: 1.125rem;
    font-weight: 600;
    color: #1e293b;
}

.vehicle-title i {
    color: #FF6B35;
    font-size: 1.25rem;
}

.tier-count {
    background: #f1f5f9;
    color: #64748b;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 500;
}

.vehicle-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.25rem;
    border-radius: 12px;
    border: none;
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: white;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.9rem;
}

.vehicle-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
}

/* Rates List */
.rates-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.rate-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1.25rem;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    transition: all 0.3s ease;
}

.rate-item:hover {
    border-color: #FF6B35;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.1);
}

.rate-info {
    display: flex;
    align-items: center;
    gap: 1rem;
    flex: 1;
}

.rate-tier {
    font-weight: 600;
    color: #1e293b;
    min-width: 80px;
}

.rate-range {
    color: #64748b;
    min-width: 120px;
}

.rate-value {
    font-weight: 600;
    color: #FF6B35;
    min-width: 120px;
}

.rate-status {
    margin: 0 1rem;
}

.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 500;
}

.status-badge.active {
    background: rgba(34, 197, 94, 0.1);
    color: #16a34a;
    border: 1px solid rgba(34, 197, 94, 0.2);
}

.status-badge.inactive {
    background: rgba(239, 68, 68, 0.1);
    color: #dc2626;
    border: 1px solid rgba(239, 68, 68, 0.2);
}

.rate-actions {
    display: flex;
    gap: 0.5rem;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.rate-item:hover .rate-actions {
    opacity: 1;
}

.rate-btn {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.9rem;
}

.rate-btn.edit {
    background: rgba(59, 130, 246, 0.1);
    color: #2563eb;
}

.rate-btn.edit:hover {
    background: #2563eb;
    color: white;
    transform: scale(1.1);
}

.rate-btn.toggle {
    background: rgba(107, 114, 128, 0.1);
    color: #6b7280;
}

.rate-btn.toggle:hover {
    background: #6b7280;
    color: white;
    transform: scale(1.1);
}

.rate-btn.delete {
    background: rgba(239, 68, 68, 0.1);
    color: #dc2626;
}

.rate-btn.delete:hover {
    background: #dc2626;
    color: white;
    transform: scale(1.1);
}

/* Commission Preview */
.commission-preview {
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    padding: 1rem;
}

.preview-header {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #64748b;
    font-size: 0.9rem;
    margin-bottom: 0.5rem;
}

.preview-result {
    font-size: 0.9rem;
    color: #1e293b;
}

.loading-text {
    color: #9ca3af;
    font-style: italic;
}

/* Empty State */
.empty-state {
    background: white;
    border-radius: 16px;
    padding: 4rem 2rem;
    text-align: center;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
    border: 1px solid #e2e8f0;
}

.empty-icon {
    font-size: 4rem;
    color: #e5e7eb;
    margin-bottom: 1.5rem;
}

.empty-text h3 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #64748b;
    margin-bottom: 0.5rem;
}

.empty-text p {
    color: #9ca3af;
    margin-bottom: 2rem;
}

.empty-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem 2rem;
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    color: white;
    border-radius: 12px;
    text-decoration: none;
    font-weight: 500;
    transition: all 0.3s ease;
}

.empty-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 53, 0.3);
}

/* Modal Styles */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.6);
    z-index: 9999;
    display: flex;
    align-items: center;
    justify-content: center;
    backdrop-filter: blur(4px);
}

.modal-content {
    background: white;
    border-radius: 16px;
    box-shadow: 0 20px 25px rgba(0, 0, 0, 0.1);
    max-width: 600px;
    width: 90%;
    max-height: 90vh;
    overflow-y: auto;
    animation: modalSlideIn 0.3s ease-out;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateY(-20px) scale(0.95);
    }
    to {
        opacity: 1;
        transform: translateY(0) scale(1);
    }
}

.modal-header {
    padding: 1.5rem 2rem;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.modal-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #1e293b;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.modal-title i {
    color: #FF6B35;
}

.modal-close {
    background: none;
    border: none;
    color: #64748b;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.modal-close:hover {
    background: #f1f5f9;
    color: #1e293b;
}

.modal-body {
    padding: 2rem;
}

/* Calculator Form */
.calculator-form {
    margin-bottom: 2rem;
}

.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr 1fr;
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-label {
    font-weight: 500;
    color: #374151;
    font-size: 0.95rem;
}

.form-input {
    padding: 0.875rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
    font-size: 0.95rem;
    transition: all 0.3s ease;
    background: white;
    font-family: inherit;
}

.form-input:focus {
    outline: none;
    border-color: #FF6B35;
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    padding: 0.875rem 1.5rem;
    border: none;
    border-radius: 12px;
    font-family: inherit;
    font-weight: 500;
    font-size: 0.95rem;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    color: white;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 53, 0.4);
}

.full-width {
    width: 100%;
}

.calculator-result {
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    padding: 1.5rem;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 1.5rem 1rem;
    }
    
    .header-info {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .header-actions {
        width: 100%;
        justify-content: stretch;
    }
    
    .action-btn {
        flex: 1;
        justify-content: center;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
        padding: 0 1rem;
    }
    
    .rates-container {
        padding: 0 1rem 2rem;
    }
    
    .filter-section {
        padding: 0 1rem;
    }
    
    .filter-row {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .warehouse-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .warehouse-actions {
        width: 100%;
        justify-content: stretch;
    }
    
    .warehouse-btn {
        flex: 1;
        justify-content: center;
    }
    
    .vehicle-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .rate-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .rate-actions {
        opacity: 1;
        width: 100%;
        justify-content: center;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .modal-content {
        width: 95%;
        margin: 1rem;
    }
    
    .modal-header,
    .modal-body {
        padding: 1rem;
    }
    
    .title-text h1 {
        font-size: 1.5rem;
    }
}

/* Animation */
@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.warehouse-card {
    animation: fadeIn 0.5s ease-out;
}

.warehouse-card:nth-child(1) { animation-delay: 0.1s; }
.warehouse-card:nth-child(2) { animation-delay: 0.2s; }
.warehouse-card:nth-child(3) { animation-delay: 0.3s; }
.warehouse-card:nth-child(4) { animation-delay: 0.4s; }
</style>

<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize filter toggle
    window.toggleFilters = function() {
        const content = document.getElementById('filterContent');
        const toggle = document.querySelector('.filter-toggle i');
        
        if (content.style.display === 'none') {
            content.style.display = 'block';
            toggle.style.transform = 'rotate(180deg)';
        } else {
            content.style.display = 'none';
            toggle.style.transform = 'rotate(0deg)';
        }
    };

    // Calculator functions
    window.showCalculator = function() {
        document.getElementById('calculatorModal').style.display = 'flex';
        document.body.style.overflow = 'hidden';
    };

    window.closeCalculator = function() {
        document.getElementById('calculatorModal').style.display = 'none';
        document.body.style.overflow = '';
        document.getElementById('calculatorResult').style.display = 'none';
    };

    // Filter functions
    window.clearFilters = function() {
        window.location.href = '{{ route("admin.commission-rates.index") }}';
    };

    window.filterByStatus = function(status) {
        const url = new URL(window.location);
        url.searchParams.set('status', status);
        window.location.href = url.toString();
    };

    // Rate management functions
    window.addRateForWarehouse = function(warehouseCode) {
        window.location.href = `{{ route('admin.commission-rates.create') }}?warehouse=${warehouseCode}`;
    };

    window.editRate = function(rateId) {
        window.location.href = `/admin/commission-rates/${rateId}/edit`;
    };

    window.toggleRateStatus = function(rateId) {
        Swal.fire({
            title: 'ยืนยันการเปลี่ยนสถานะ',
            text: 'คุณต้องการเปลี่ยนสถานะการใช้งานของอัตรานี้หรือไม่?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#FF6B35',
            cancelButtonColor: '#6B7280',
            confirmButtonText: 'ยืนยัน',
            cancelButtonText: 'ยกเลิก'
        }).then((result) => {
            if (result.isConfirmed) {
                // Simulate API call
                Swal.fire({
                    icon: 'success',
                    title: 'สำเร็จ!',
                    text: 'เปลี่ยนสถานะเรียบร้อยแล้ว',
                    confirmButtonColor: '#FF6B35'
                }).then(() => {
                    location.reload();
                });
            }
        });
    };

    window.deleteRate = function(rateId) {
        Swal.fire({
            title: 'ยืนยันการลบ',
            text: 'คุณต้องการลบอัตราค่าคอมมิชชั่นนี้หรือไม่? การดำเนินการนี้ไม่สามารถยกเลิกได้',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#EF4444',
            cancelButtonColor: '#6B7280',
            confirmButtonText: 'ลบ',
            cancelButtonText: 'ยกเลิก'
        }).then((result) => {
            if (result.isConfirmed) {
                // Simulate API call
                Swal.fire({
                    icon: 'success',
                    title: 'สำเร็จ!',
                    text: 'ลบอัตราเรียบร้อยแล้ว',
                    confirmButtonColor: '#FF6B35'
                }).then(() => {
                    location.reload();
                });
            }
        });
    };

    window.previewCommission = function(warehouseCode, vehicleType) {
        document.getElementById('calc_warehouse').value = warehouseCode;
        document.getElementById('calc_vehicle').value = vehicleType;
        document.getElementById('calc_packages').value = 150;
        showCalculator();
    };

    window.bulkAction = function(warehouseCode) {
        Swal.fire({
            title: 'เลือกการดำเนินการ',
            text: `จัดการอัตราทั้งหมดของคลัง ${warehouseCode}`,
            icon: 'question',
            showCancelButton: true,
            showDenyButton: true,
            confirmButtonText: 'เปิดใช้งานทั้งหมด',
            denyButtonText: 'ปิดใช้งานทั้งหมด',
            cancelButtonText: 'ยกเลิก',
            confirmButtonColor: '#22C55E',
            denyButtonColor: '#EF4444',
            cancelButtonColor: '#6B7280'
        }).then((result) => {
            if (result.isConfirmed || result.isDenied) {
                const action = result.isConfirmed ? 'เปิดใช้งาน' : 'ปิดใช้งาน';
                Swal.fire({
                    icon: 'success',
                    title: 'สำเร็จ!',
                    text: `${action}อัตราทั้งหมดเรียบร้อยแล้ว`,
                    confirmButtonColor: '#FF6B35'
                }).then(() => {
                    location.reload();
                });
            }
        });
    };

    window.exportRates = function() {
        Swal.fire({
            title: 'กำลังเตรียมข้อมูล',
            text: 'กรุณารอสักครู่...',
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });
        
        setTimeout(() => {
            Swal.close();
            const toast = Swal.mixin({
                toast: true,
                position: 'top-end',
                showConfirmButton: false,
                timer: 3000
            });
            
            toast.fire({
                icon: 'success',
                title: 'ส่งออกข้อมูลสำเร็จ!'
            });
        }, 2000);
    };

    window.showWarehouses = function() {
        window.location.href = '{{ route("admin.warehouses.index") }}';
    };

    window.showAnalytics = function() {
        Swal.fire({
            icon: 'info',
            title: 'ฟีเจอร์วิเคราะห์',
            text: 'ฟีเจอร์นี้จะเปิดให้ใช้งานในเร็วๆ นี้',
            confirmButtonColor: '#FF6B35'
        });
    };

    // Calculator form submission
    document.getElementById('calculatorForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = Object.fromEntries(formData);
        
        if (!data.warehouse_code || !data.vehicle_type || !data.package_count) {
            Swal.fire({
                icon: 'warning',
                title: 'กรอกข้อมูลไม่ครบ',
                text: 'กรุณากรอกข้อมูลให้ครบถ้วน',
                confirmButtonColor: '#FF6B35'
            });
            return;
        }
        
        // Show loading
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> กำลังคำนวณ...';
        submitBtn.disabled = true;
        
        // Simulate calculation
        setTimeout(() => {
            const result = {
                package_count: data.package_count,
                total_commission: (parseFloat(data.package_count) * 2.5).toFixed(2),
                average_per_package: '2.50',
                breakdown: [
                    { tier: 1, range: '1-50', packages_in_tier: Math.min(50, data.package_count), rate_per_package: '2.00', commission: (Math.min(50, data.package_count) * 2.00).toFixed(2) },
                    { tier: 2, range: '51-100', packages_in_tier: Math.max(0, Math.min(50, data.package_count - 50)), rate_per_package: '2.50', commission: (Math.max(0, Math.min(50, data.package_count - 50)) * 2.50).toFixed(2) },
                    { tier: 3, range: '101+', packages_in_tier: Math.max(0, data.package_count - 100), rate_per_package: '3.00', commission: (Math.max(0, data.package_count - 100) * 3.00).toFixed(2) }
                ].filter(tier => tier.packages_in_tier > 0)
            };
            
            displayCalculationResult(result);
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }, 1500);
    });

    function displayCalculationResult(data) {
        const resultDiv = document.getElementById('calculatorResult');
        
        let breakdownHtml = data.breakdown.map(tier => `
            <div class="rate-item">
                <div class="rate-info">
                    <div class="rate-tier">ขั้นที่ ${tier.tier}</div>
                    <div class="rate-range">${tier.range} ชิ้น</div>
                    <div class="rate-value">${tier.packages_in_tier} × ${tier.rate_per_package} = ${tier.commission} บาท</div>
                </div>
            </div>
        `).join('');
        
        resultDiv.innerHTML = `
            <div class="commission-preview">
                <div class="preview-header">
                    <i class="fas fa-calculator"></i>
                    <span>ผลการคำนวณค่าคอมมิชชั่น</span>
                </div>
                
                <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 1rem; margin: 1rem 0;">
                    <div style="background: rgba(59, 130, 246, 0.1); padding: 1rem; border-radius: 8px; text-align: center;">
                        <div style="font-size: 1.5rem; font-weight: 600; color: #2563eb;">${data.package_count}</div>
                        <div style="font-size: 0.85rem; color: #2563eb;">ชิ้น</div>
                    </div>
                    <div style="background: rgba(34, 197, 94, 0.1); padding: 1rem; border-radius: 8px; text-align: center;">
                        <div style="font-size: 1.5rem; font-weight: 600; color: #16a34a;">${data.total_commission}</div>
                        <div style="font-size: 0.85rem; color: #16a34a;">บาท รวม</div>
                    </div>
                    <div style="background: rgba(255, 107, 53, 0.1); padding: 1rem; border-radius: 8px; text-align: center;">
                        <div style="font-size: 1.5rem; font-weight: 600; color: #FF6B35;">${data.average_per_package}</div>
                        <div style="font-size: 0.85rem; color: #FF6B35;">บาท/ชิ้น</div>
                    </div>
                </div>
                
                <div style="margin-top: 1rem;">
                    <h4 style="margin-bottom: 1rem; color: #1e293b;">รายละเอียดการคำนวณ:</h4>
                    ${breakdownHtml}
                </div>
            </div>
        `;
        
        resultDiv.style.display = 'block';
    }

    // Load commission previews
    document.querySelectorAll('[id^="preview-"]').forEach(previewElement => {
        setTimeout(() => {
            const mockCommission = (Math.random() * 300 + 200).toFixed(2);
            const mockAverage = (mockCommission / 150).toFixed(2);
            
            previewElement.innerHTML = `
                <div style="display: flex; align-items: center; justify-content: space-between;">
                    <span>ค่าคอมมิชชั่น: <strong style="color: #FF6B35;">${mockCommission} บาท</strong></span>
                    <span style="font-size: 0.85rem; color: #64748b;">(เฉลี่ย ${mockAverage} บาท/ชิ้น)</span>
                </div>
            `;
        }, Math.random() * 2000 + 500);
    });

    // Close modals on ESC key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeCalculator();
        }
    });

    // Show success/error messages from session
    @if(session('success'))
        const toast = Swal.mixin({
            toast: true,
            position: 'top-end',
            showConfirmButton: false,
            timer: 4000,
            timerProgressBar: true
        });
        
        toast.fire({
            icon: 'success',
            title: '{{ session('success') }}'
        });
    @endif

    @if(session('error'))
        Swal.fire({
            icon: 'error',
            title: 'เกิดข้อผิดพลาด!',
            text: '{{ session('error') }}',
            confirmButtonColor: '#FF6B35'
        });
    @endif
});
</script>
@endsection