@extends('layouts.app')

@section('title', __('deliveries.create_delivery'))
@section('page-title', __('deliveries.create_delivery'))
@section('mobile-title', __('deliveries.new_record'))

@section('content')
<!-- Header Card -->
<div class="header-card">
    <div class="header-content">
        <div class="header-text">
            <div class="header-title">
                <i class="fas fa-plus-circle"></i>
                {{ __('deliveries.create_delivery') }}
            </div>
            <div class="header-subtitle">{{ __('deliveries.record_daily_delivery') }}</div>
        </div>
        <div class="header-actions">
            <form method="POST" action="{{ route('language.switch') }}" class="language-form">
                @csrf
                <input type="hidden" name="language" value="{{ app()->getLocale() == 'th' ? 'en' : 'th' }}">
                <button type="submit" class="language-btn">
                    <i class="fas fa-globe"></i>
                    <span class="desktop-text">{{ app()->getLocale() == 'th' ? 'EN' : 'TH' }}</span>
                </button>
            </form>
            <div class="header-icon">
                <i class="fas fa-{{ $user->role == '2wheel' ? 'motorcycle' : 'truck' }}"></i>
            </div>
        </div>
    </div>
</div>

<!-- User Info Card -->
<div class="user-info-card">
    <div class="info-header">
        <div class="info-title">
            <i class="fas fa-user"></i>
            {{ __('deliveries.user_info') }}
        </div>
    </div>
    <div class="info-grid">
        <div class="info-item">
            <div class="info-label">{{ __('deliveries.full_name') }}</div>
            <div class="info-value">{{ $user->name }}</div>
        </div>
        <div class="info-item">
            <div class="info-label">{{ __('deliveries.warehouse') }}</div>
            <div class="info-value">{{ $user->warehouse->name ?? $warehouse_code ?? 'N/A' }}</div>
        </div>
        <div class="info-item">
            <div class="info-label">{{ __('deliveries.vehicle_type') }}</div>
            <div class="info-value">
                <i class="fas fa-{{ $user->role == '2wheel' ? 'motorcycle' : 'truck' }}"></i>
                {{ $user->role == '2wheel' ? __('deliveries.2_wheeler') : __('deliveries.4_wheeler') }}
            </div>
        </div>
        <div class="info-item">
            <div class="info-label">{{ __('deliveries.employee_id') }}</div>
            <div class="info-value">#{{ $user->id }}</div>
        </div>
    </div>
</div>

<!-- Delivery Form -->
<form action="{{ route('vehicle.deliveries.store') }}" method="POST" id="deliveryForm">
    @csrf
    
    <!-- Form Card -->
    <div class="form-card">
        <div class="form-header">
            <div class="form-title">
                <i class="fas fa-edit"></i>
                {{ __('deliveries.delivery_info') }}
            </div>
        </div>
        
        <div class="form-body">
            <!-- Delivery Date -->
            <div class="form-group">
                <label class="form-label required">{{ __('deliveries.delivery_date') }}</label>
                <div class="input-group">
                    <div class="input-icon">
                        <i class="fas fa-calendar-alt"></i>
                    </div>
                    <input type="date" 
                           name="delivery_date" 
                           id="delivery_date"
                           class="form-input @error('delivery_date') error @enderror" 
                           value="{{ old('delivery_date', date('Y-m-d')) }}"
                           max="{{ date('Y-m-d') }}">
                </div>
                @error('delivery_date')
                    <div class="error-text">{{ $message }}</div>
                @enderror
                <div class="form-hint">{{ __('deliveries.select_delivery_date') }}</div>
            </div>

            <!-- Parcel Count -->
            <div class="form-group">
                <label class="form-label required">{{ __('deliveries.parcel_count') }} ({{ __('deliveries.pieces') }})</label>
                <div class="input-group">
                    <div class="input-icon">
                        <i class="fas fa-boxes"></i>
                    </div>
                    <input type="number" 
                           name="parcel_count" 
                           id="parcel_count"
                           class="form-input @error('parcel_count') error @enderror" 
                           value="{{ old('parcel_count') }}"
                           min="1"
                           placeholder="{{ __('deliveries.enter_parcel_count') }}">
                </div>
                @error('parcel_count')
                    <div class="error-text">{{ $message }}</div>
                @enderror
                <div class="form-hint">{{ __('deliveries.enter_total_parcels') }}</div>
            </div>
        </div>
    </div>

    <!-- Commission Preview -->
    <div class="commission-preview" id="commissionPreview" style="display: none;">
        <div class="preview-header">
            <div class="preview-title">
                <i class="fas fa-calculator"></i>
                {{ __('deliveries.estimated_commission') }}
            </div>
        </div>
        <div class="preview-body">
            <div class="preview-loading">
                <div class="loading-spinner"></div>
                <span>{{ __('deliveries.calculating') }}</span>
            </div>
            <div class="preview-result" style="display: none;">
                <div class="commission-amount">
                    <span class="currency">฿</span>
                    <span class="amount">0</span>
                </div>
                <div class="commission-breakdown"></div>
            </div>
        </div>
    </div>
</form>

<!-- Commission Rates -->
@if(isset($commissionRates) && $commissionRates->count() > 0)
<div class="rates-card">
    <div class="rates-header">
        <div class="rates-title">
            <i class="fas fa-info-circle"></i>
            {{ __('deliveries.current_commission_rates') }}
        </div>
    </div>
    <div class="rates-list">
        @foreach($commissionRates as $rate)
            <div class="rate-item">
                <div class="rate-range">{{ $rate->range_display }}</div>
                <div class="rate-amount">฿{{ number_format($rate->rate_per_package, 2) }}/{{ __('deliveries.pieces') }}</div>
            </div>
        @endforeach
    </div>
</div>
@endif

<!-- Action Buttons -->
<div class="action-buttons">
    <button type="button" class="btn-secondary" onclick="window.history.back()">
        <i class="fas fa-arrow-left"></i>
        {{ __('deliveries.cancel') }}
    </button>
    <button type="submit" class="btn-primary" id="submitBtn" form="deliveryForm">
        <i class="fas fa-save"></i>
        {{ __('deliveries.save_data') }}
    </button>
</div>

@push('styles')
<style>
/* Header Card */
.header-card {
    background: var(--gradient);
    border-radius: 20px;
    padding: 20px;
    margin-bottom: 20px;
    color: white;
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-actions {
    display: flex;
    align-items: center;
    gap: 12px;
}

.language-form {
    margin: 0;
}

.language-btn {
    background: rgba(255,255,255,0.2);
    border: 1px solid rgba(255,255,255,0.3);
    color: white;
    padding: 8px 12px;
    border-radius: 8px;
    font-size: 12px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    display: flex;
    align-items: center;
    gap: 4px;
    backdrop-filter: blur(10px);
}

.language-btn:hover {
    background: rgba(255,255,255,0.3);
    transform: translateY(-1px);
}

.language-btn:active {
    transform: translateY(0);
}

@media (max-width: 768px) {
    .language-btn .desktop-text {
        display: none;
    }
    
    .language-btn {
        width: 36px;
        height: 36px;
        padding: 0;
        border-radius: 50%;
        justify-content: center;
    }
}

.header-title {
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 4px;
    display: flex;
    align-items: center;
    gap: 8px;
}

.header-subtitle {
    font-size: 14px;
    opacity: 0.9;
}

.header-icon {
    width: 60px;
    height: 60px;
    background: rgba(255,255,255,0.2);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
}

/* User Info Card */
.user-info-card {
    background: white;
    border-radius: 16px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
}

.info-header {
    margin-bottom: 16px;
}

.info-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--gray-900);
    display: flex;
    align-items: center;
    gap: 8px;
}

.info-title i {
    color: var(--primary);
}

.info-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 16px;
}

.info-item {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.info-label {
    font-size: 12px;
    color: var(--gray-500);
    font-weight: 500;
}

.info-value {
    font-size: 14px;
    font-weight: 600;
    color: var(--gray-900);
    display: flex;
    align-items: center;
    gap: 6px;
}

.info-value i {
    color: var(--primary);
}

/* Form Card */
.form-card {
    background: white;
    border-radius: 16px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
    overflow: hidden;
}

.form-header {
    background: var(--gradient-soft);
    padding: 16px 20px;
    border-bottom: 1px solid rgba(255,107,53,0.1);
}

.form-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary);
    display: flex;
    align-items: center;
    gap: 8px;
}

.form-body {
    padding: 20px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--gray-900);
    margin-bottom: 8px;
}

.form-label.required::after {
    content: ' *';
    color: #EF4444;
}

.input-group {
    position: relative;
}

.input-icon {
    position: absolute;
    left: 16px;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray-400);
    z-index: 2;
}

.form-input {
    width: 100%;
    padding: 16px 16px 16px 48px;
    border: 2px solid var(--gray-200);
    border-radius: 12px;
    font-size: 16px;
    font-family: inherit;
    transition: all 0.3s;
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(255,107,53,0.1);
}

.form-input.error {
    border-color: #EF4444;
}

.error-text {
    color: #EF4444;
    font-size: 12px;
    margin-top: 4px;
    display: flex;
    align-items: center;
    gap: 4px;
}

.form-hint {
    color: var(--gray-500);
    font-size: 12px;
    margin-top: 4px;
}

/* Commission Preview */
.commission-preview {
    background: white;
    border-radius: 16px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(59,130,246,0.06);
    border: 2px solid #3B82F6;
    overflow: hidden;
}

.preview-header {
    background: rgba(59,130,246,0.05);
    padding: 16px 20px;
    border-bottom: 1px solid rgba(59,130,246,0.1);
}

.preview-title {
    font-size: 16px;
    font-weight: 600;
    color: #3B82F6;
    display: flex;
    align-items: center;
    gap: 8px;
}

.preview-body {
    padding: 20px;
}

.preview-loading {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 12px;
    color: #3B82F6;
    font-size: 14px;
}

.loading-spinner {
    width: 20px;
    height: 20px;
    border: 2px solid rgba(59,130,246,0.2);
    border-top-color: #3B82F6;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    to { transform: rotate(360deg); }
}

.commission-amount {
    text-align: center;
    margin-bottom: 16px;
}

.commission-amount .currency {
    font-size: 20px;
    color: #3B82F6;
    font-weight: 600;
}

.commission-amount .amount {
    font-size: 32px;
    font-weight: 700;
    color: #3B82F6;
    margin-left: 4px;
}

.commission-breakdown {
    font-size: 12px;
    color: var(--gray-600);
    text-align: center;
}

/* Rates Card */
.rates-card {
    background: white;
    border-radius: 16px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
    overflow: hidden;
}

.rates-header {
    background: var(--gradient-soft);
    padding: 16px 20px;
    border-bottom: 1px solid rgba(255,107,53,0.1);
}

.rates-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary);
    display: flex;
    align-items: center;
    gap: 8px;
}

.rates-list {
    padding: 16px 20px;
}

.rate-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid var(--gray-100);
}

.rate-item:last-child {
    border-bottom: none;
}

.rate-range {
    font-size: 14px;
    color: var(--gray-700);
    font-weight: 500;
}

.rate-amount {
    font-size: 14px;
    font-weight: 600;
    color: var(--primary);
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 12px;
    margin-bottom: 100px;
}

.btn-primary,
.btn-secondary {
    flex: 1;
    padding: 16px 20px;
    border-radius: 12px;
    font-size: 16px;
    font-weight: 600;
    border: none;
    cursor: pointer;
    transition: all 0.3s;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.btn-primary {
    background: var(--gradient);
    color: white;
    box-shadow: 0 4px 12px rgba(255,107,53,0.3);
}

.btn-primary:active {
    transform: scale(0.98);
}

.btn-primary.disabled {
    pointer-events: none;
    opacity: 0.5 !important;
    cursor: not-allowed;
}

.btn-secondary {
    background: var(--gray-100);
    color: var(--gray-700);
    border: 2px solid var(--gray-200);
}

.btn-secondary:active {
    transform: scale(0.98);
    background: var(--gray-200);
}

/* Desktop Styles */
@media (min-width: 769px) {
    .header-card {
        display: none;
    }
    
    .info-grid {
        grid-template-columns: repeat(4, 1fr);
    }
    
    .action-buttons {
        justify-content: flex-end;
    }
    
    .btn-primary,
    .btn-secondary {
        flex: none;
        min-width: 140px;
    }
}

@media (max-width: 380px) {
    .info-grid {
        grid-template-columns: 1fr;
    }
    
    .action-buttons {
        flex-direction: column;
    }
}



.page-header::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 200px;
    height: 200px;
    background: radial-gradient(circle, rgba(255, 255, 255, 0.1) 0%, transparent 70%);
    border-radius: 50%;
    transform: translate(50%, -50%);
}

.page-header::after {
    content: '';
    position: absolute;
    bottom: 0;
    left: 0;
    width: 150px;
    height: 150px;
    background: radial-gradient(circle, rgba(255, 255, 255, 0.08) 0%, transparent 70%);
    border-radius: 50%;
    transform: translate(-30%, 30%);
}

.page-title {
    font-size: 1.75rem;
    font-weight: 700;
    color: var(--white);
    margin: 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    position: relative;
    z-index: 2;
}

.page-subtitle {
    font-size: 1rem;
    color: rgba(255, 255, 255, 0.9);
    margin: 0.5rem 0 0 0;
    font-weight: 400;
    position: relative;
    z-index: 2;
}

.form-container {
    display: grid;
    grid-template-columns: 1fr minmax(300px, 400px);
    gap: 1.5rem;
    align-items: start;
}

.form-column {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.info-card,
.form-card,
.rates-info-card {
    background: var(--white);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-sm);
    overflow: hidden;
}

.info-card-header,
.form-card-header,
.rates-header {
    background: linear-gradient(135deg, rgba(255, 107, 53, 0.05) 0%, rgba(255, 140, 97, 0.05) 100%);
    padding: 0.875rem 1.25rem;
    font-weight: 600;
    font-size: 1.05rem;
    color: var(--dark-gray);
    border-bottom: 1px solid var(--medium-gray);
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.info-card-body,
.form-card-body {
    padding: 1.25rem;
}

.info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.info-item {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.info-label {
    font-size: 0.85rem;
    color: var(--gray);
}

.info-value {
    font-weight: 500;
    color: var(--dark-gray);
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 500;
    color: var(--dark-gray);
}

.form-label.required::after {
    content: ' *';
    color: var(--danger);
}

.input-with-icon {
    position: relative;
}

.input-with-icon i {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray);
}

.input-with-icon .form-control {
    padding-left: 2.75rem;
}

.form-control {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid var(--medium-gray);
    border-radius: var(--radius-md);
    font-family: inherit;
    font-size: 1rem;
    transition: var(--transition-fast);
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-orange);
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.form-control.is-invalid {
    border-color: var(--danger);
}

.error-message {
    display: block;
    color: var(--danger);
    font-size: 0.875rem;
    margin-top: 0.25rem;
}

.form-hint {
    display: block;
    color: var(--gray);
    font-size: 0.875rem;
    margin-top: 0.25rem;
}

.commission-preview {
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 2px solid #60a5fa;
    border-radius: var(--radius-md);
    padding: 1.5rem;
    margin-top: 1rem;
}

.preview-header {
    font-weight: 600;
    color: #2563eb;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.preview-loading {
    text-align: center;
    color: #60a5fa;
    padding: 1rem;
}

.preview-amount {
    text-align: center;
    margin-bottom: 1rem;
}

.preview-amount .currency {
    font-size: 1.5rem;
    color: #2563eb;
}

.preview-amount .amount {
    font-size: 2.5rem;
    font-weight: 700;
    color: #2563eb;
}

.preview-breakdown {
    font-size: 0.875rem;
    color: #64748b;
}

.rates-info-card {
    margin-top: 1rem;
}

.rates-table {
    padding: 0.75rem;
    overflow-x: auto;
}

.rates-table table {
    width: 100%;
    border-collapse: collapse;
    font-size: 0.9rem;
}

.rates-table th {
    background: var(--light-gray);
    padding: 0.6rem;
    text-align: left;
    font-weight: 600;
    color: var(--dark-gray);
    font-size: 0.85rem;
}

.rates-table td {
    padding: 0.6rem;
    border-top: 1px solid var(--medium-gray);
}




</style>
@endpush

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const parcelInput = document.getElementById('parcel_count');
    const dateInput = document.getElementById('delivery_date');
    const previewDiv = document.getElementById('commissionPreview');
    const form = document.getElementById('deliveryForm');
    const submitBtn = document.getElementById('submitBtn');
    let calculationTimeout;

    // Remove any disabled state
    submitBtn.classList.remove('disabled');
    submitBtn.style.opacity = '1';

    // Validation and Commission Preview
    parcelInput.addEventListener('input', function() {
        const parcelValue = this.value;
        const parcelNumber = parseInt(parcelValue);
        
        // Show error if exactly 0
        if (parcelValue !== '' && parcelNumber === 0) {
            if (window.Swal) {
                Swal.fire({
                    title: '{{ __('deliveries.error') }}!',
                    text: '{{ __('deliveries.parcel_count_must_be_greater_than_zero') }}',
                    icon: 'error',
                    confirmButtonColor: '#FF6B35',
                    confirmButtonText: '{{ __('deliveries.ok') }}'
                });
            }
            this.value = '';
            return;
        }
        
        clearTimeout(calculationTimeout);
        const count = parseInt(this.value);
        if (count > 0) {
            previewDiv.style.display = 'block';
            document.querySelector('.preview-loading').style.display = 'flex';
            document.querySelector('.preview-result').style.display = 'none';
            calculationTimeout = setTimeout(() => {
                calculateCommission(count);
            }, 500);
        } else {
            previewDiv.style.display = 'none';
        }
    });

    function calculateCommission(parcelCount) {
        // Simulate commission calculation
        setTimeout(() => {
            document.querySelector('.preview-loading').style.display = 'none';
            document.querySelector('.preview-result').style.display = 'block';
            
            // Simple calculation (replace with actual API call)
            const commission = parcelCount * 30; // 30 baht per parcel
            document.querySelector('.commission-amount .amount').textContent = 
                new Intl.NumberFormat('th-TH').format(commission);
            document.querySelector('.commission-breakdown').innerHTML = 
                `${parcelCount} {{ __('deliveries.pieces') }} × ฿30 = ฿${new Intl.NumberFormat('th-TH').format(commission)}`;
        }, 1000);
    }

    // Form submission with confirmation
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        const parcelCount = parcelInput.value;
        const deliveryDate = dateInput.value;
        const commission = document.querySelector('.commission-amount .amount')?.textContent || "0";
        
        if (window.Swal) {
            Swal.fire({
                title: '{{ __('deliveries.confirm_save') }}',
                html: `<div style="text-align: left;">
                    <p><strong>{{ __('deliveries.delivery_date') }}:</strong> ${deliveryDate}</p>
                    <p><strong>{{ __('deliveries.parcel_count') }}:</strong> ${parcelCount} {{ __('deliveries.pieces') }}</p>
                    <p><strong>{{ __('deliveries.commission') }}:</strong> ฿${commission}</p>
                </div>`,
                icon: 'question',
                showCancelButton: true,
                confirmButtonColor: '#FF6B35',
                cancelButtonColor: '#94a3b8',
                confirmButtonText: '{{ __('deliveries.save') }}',
                cancelButtonText: '{{ __('deliveries.cancel') }}'
            }).then((result) => {
                if (result.isConfirmed) {
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> {{ __('deliveries.saving') }}...';
                    submitBtn.classList.add('disabled');
                    form.submit();
                }
            });
        } else {
            if (confirm('{{ __('deliveries.confirm_save_data') }}')) {
                form.submit();
            }
        }
    });

    // Error messages
    @if($errors->any())
        if (window.Swal) {
            Swal.fire({
                title: '{{ __('deliveries.error_occurred') }}!',
                html: '@foreach($errors->all() as $error)<p>{{ $error }}</p>@endforeach',
                icon: 'error',
                confirmButtonColor: '#FF6B35',
                confirmButtonText: '{{ __('deliveries.ok') }}'
            });
        }
    @endif
});
</script>
@endpush

@endsection