<?php
/**
 * Hardcoded String Scanner
 * Scans for Thai/English hardcoded strings in Blade templates and PHP files
 */

require_once __DIR__ . '/../vendor/autoload.php';

use Illuminate\Support\Facades\File;

class HardcodedStringScanner
{
    private $patterns = [
        // Thai text patterns
        'thai' => '/[\u{0E00}-\u{0E7F}]+/u',
        // English text in quotes (excluding __() calls)
        'english_quotes' => '/(?<!__)(["\'])([A-Za-z][A-Za-z0-9\s\-_.,!?]+)\1/',
        // HTML title/alt attributes with text
        'html_attributes' => '/(title|alt|placeholder)=["\']([A-Za-z\u{0E00}-\u{0E7F}][^"\']*)["\']/',
        // JavaScript strings
        'js_strings' => '/(?:alert|confirm|console\.log)\s*\(\s*["\']([^"\']+)["\']\s*\)/',
    ];
    
    private $excludePatterns = [
        // Exclude translation calls
        '/__\s*\(\s*["\'][^"\']+["\']\s*\)/',
        // Exclude config calls
        '/config\s*\(\s*["\'][^"\']+["\']\s*\)/',
        // Exclude route calls
        '/route\s*\(\s*["\'][^"\']+["\']\s*\)/',
        // Exclude CSS classes and IDs
        '/class\s*=\s*["\'][^"\']*["\']/',
        '/id\s*=\s*["\'][^"\']*["\']/',
        // Exclude URLs and paths
        '/https?:\/\/[^\s"\']+/',
        '/\/[a-zA-Z0-9\/_-]+/',
        // Exclude variable names and function names
        '/\$[a-zA-Z_][a-zA-Z0-9_]*/',
        '/[a-zA-Z_][a-zA-Z0-9_]*\s*\(/',
    ];
    
    private $results = [];
    
    public function scan($directory)
    {
        $this->scanDirectory($directory);
        return $this->results;
    }
    
    private function scanDirectory($directory)
    {
        $files = File::allFiles($directory);
        
        foreach ($files as $file) {
            $extension = $file->getExtension();
            
            // Only scan relevant file types
            if (!in_array($extension, ['php', 'blade.php', 'js', 'vue'])) {
                continue;
            }
            
            $this->scanFile($file);
        }
    }
    
    private function scanFile($file)
    {
        $content = File::get($file);
        $lines = explode("\n", $content);
        $filePath = str_replace(base_path(), '', $file->getPathname());
        
        foreach ($lines as $lineNumber => $line) {
            $this->scanLine($filePath, $lineNumber + 1, $line);
        }
    }
    
    private function scanLine($filePath, $lineNumber, $line)
    {
        // Skip if line is already using translation
        if (preg_match('/__\s*\(/', $line)) {
            return;
        }
        
        // Check for Thai text
        if (preg_match($this->patterns['thai'], $line, $matches)) {
            $this->addResult($filePath, $lineNumber, $line, 'thai', $matches[0]);
        }
        
        // Check for English strings in quotes
        if (preg_match_all($this->patterns['english_quotes'], $line, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $text = $match[2];
                
                // Skip if it's likely a CSS class, ID, or technical term
                if ($this->shouldExclude($text, $line)) {
                    continue;
                }
                
                // Only include if it looks like user-facing text
                if ($this->isUserFacingText($text)) {
                    $this->addResult($filePath, $lineNumber, $line, 'english', $text);
                }
            }
        }
        
        // Check HTML attributes
        if (preg_match_all($this->patterns['html_attributes'], $line, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $text = $match[2];
                if ($this->isUserFacingText($text)) {
                    $this->addResult($filePath, $lineNumber, $line, 'html_attribute', $text);
                }
            }
        }
        
        // Check JavaScript strings
        if (preg_match_all($this->patterns['js_strings'], $line, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $text = $match[1];
                if ($this->isUserFacingText($text)) {
                    $this->addResult($filePath, $lineNumber, $line, 'javascript', $text);
                }
            }
        }
    }
    
    private function shouldExclude($text, $line)
    {
        // Check exclude patterns
        foreach ($this->excludePatterns as $pattern) {
            if (preg_match($pattern, $line)) {
                return true;
            }
        }
        
        // Exclude technical terms
        $technicalTerms = [
            'utf-8', 'charset', 'viewport', 'content', 'width', 'height',
            'javascript', 'css', 'html', 'json', 'xml', 'http', 'https',
            'admin', 'user', 'id', 'name', 'email', 'password', 'token',
            'csrf', 'method', 'post', 'get', 'put', 'delete', 'patch',
            'true', 'false', 'null', 'undefined', 'var', 'let', 'const',
            'function', 'return', 'if', 'else', 'for', 'while', 'switch',
            'case', 'break', 'continue', 'try', 'catch', 'finally',
        ];
        
        return in_array(strtolower($text), $technicalTerms);
    }
    
    private function isUserFacingText($text)
    {
        // Must be at least 2 characters
        if (strlen($text) < 2) {
            return false;
        }
        
        // Must contain at least one letter
        if (!preg_match('/[a-zA-Z\u{0E00}-\u{0E7F}]/u', $text)) {
            return false;
        }
        
        // Exclude single words that are likely technical
        if (!preg_match('/\s/', $text) && strlen($text) < 4) {
            return false;
        }
        
        // Exclude hex colors, IDs, etc.
        if (preg_match('/^[a-f0-9]{3,}$/i', $text)) {
            return false;
        }
        
        // Exclude file extensions
        if (preg_match('/\.(js|css|php|html|json|xml|png|jpg|gif|svg)$/i', $text)) {
            return false;
        }
        
        return true;
    }
    
    private function addResult($filePath, $lineNumber, $line, $type, $text)
    {
        $this->results[] = [
            'file' => $filePath,
            'line' => $lineNumber,
            'code' => trim($line),
            'type' => $type,
            'text' => $text,
            'suggested_key' => $this->suggestTranslationKey($text, $filePath),
        ];
    }
    
    private function suggestTranslationKey($text, $filePath)
    {
        // Determine domain from file path
        $domain = 'common';
        if (strpos($filePath, 'admin') !== false) {
            $domain = 'admin';
        } elseif (strpos($filePath, 'finance') !== false) {
            $domain = 'finance';
        } elseif (strpos($filePath, 'delivery') !== false || strpos($filePath, 'vehicle') !== false) {
            $domain = 'delivery';
        }
        
        // Generate key from text
        $key = strtolower($text);
        $key = preg_replace('/[^\w\s\u{0E00}-\u{0E7F}]/u', '', $key);
        $key = preg_replace('/\s+/', '_', $key);
        $key = substr($key, 0, 50); // Limit length
        
        return "{$domain}.{$key}";
    }
    
    public function generateReport()
    {
        $report = "# Hardcoded Strings Report\n\n";
        $report .= "Found " . count($this->results) . " hardcoded strings:\n\n";
        
        $groupedResults = [];
        foreach ($this->results as $result) {
            $groupedResults[$result['file']][] = $result;
        }
        
        foreach ($groupedResults as $file => $results) {
            $report .= "## {$file}\n\n";
            
            foreach ($results as $result) {
                $report .= "**Line {$result['line']}** ({$result['type']}):\n";
                $report .= "```\n{$result['code']}\n```\n";
                $report .= "**Text:** `{$result['text']}`\n";
                $report .= "**Suggested key:** `{$result['suggested_key']}`\n\n";
            }
        }
        
        return $report;
    }
    
    public function generateFixScript()
    {
        $script = "#!/bin/bash\n";
        $script .= "# Auto-generated script to fix hardcoded strings\n\n";
        
        foreach ($this->results as $result) {
            $escapedText = addslashes($result['text']);
            $translationCall = "{{ __('{$result['suggested_key']}') }}";
            
            $script .= "# Fix: {$result['file']}:{$result['line']}\n";
            $script .= "# Replace: {$escapedText}\n";
            $script .= "# With: {$translationCall}\n\n";
        }
        
        return $script;
    }
}

// Run the scanner
echo "🔍 Scanning for hardcoded strings...\n";

$scanner = new HardcodedStringScanner();

// Scan views
$scanner->scan(resource_path('views'));

// Scan controllers
$scanner->scan(app_path('Http/Controllers'));

$results = $scanner->scan('.');

echo "✅ Scan complete!\n";
echo "Found " . count($results) . " hardcoded strings.\n\n";

// Generate report
$report = $scanner->generateReport();
file_put_contents(__DIR__ . '/../HARDCODED_STRINGS_REPORT.md', $report);
echo "📄 Report saved to HARDCODED_STRINGS_REPORT.md\n";

// Generate fix script
$fixScript = $scanner->generateFixScript();
file_put_contents(__DIR__ . '/fix-hardcoded-strings.sh', $fixScript);
echo "🔧 Fix script saved to scripts/fix-hardcoded-strings.sh\n";

// Display summary
echo "\n📊 Summary by type:\n";
$types = [];
foreach ($results as $result) {
    $types[$result['type']] = ($types[$result['type']] ?? 0) + 1;
}

foreach ($types as $type => $count) {
    echo "  {$type}: {$count}\n";
}

echo "\n🎯 Top files with hardcoded strings:\n";
$files = [];
foreach ($results as $result) {
    $files[$result['file']] = ($files[$result['file']] ?? 0) + 1;
}

arsort($files);
$topFiles = array_slice($files, 0, 10, true);
foreach ($topFiles as $file => $count) {
    echo "  {$file}: {$count}\n";
}

echo "\n💡 Next steps:\n";
echo "1. Review HARDCODED_STRINGS_REPORT.md\n";
echo "2. Add missing keys to translation files\n";
echo "3. Replace hardcoded strings with __() calls\n";
echo "4. Run php artisan i18n:check to verify\n";